# Tenant-Aware Navigation System

## Overview
This implementation provides automatic tenant parameter preservation across all navigation in the admin dashboard, ensuring tenant context is never lost during navigation.

## Key Components

### 1. Utility Module (`src/utils/tenantNavigation.ts`)
Central utility for tenant-aware navigation:

```typescript
// Get tenant-aware URL with automatic parameter preservation
getTenantAwareHref(path: string): string

// Get current tenant from URL
getCurrentTenant(): string | null

// Navigate with tenant parameter
navigateWithTenant(navigate, path: string): void
```

### 2. How It Works

#### URL Detection
```javascript
const params = new URLSearchParams(window.location.search);
const tenant = params.get('tenant');
```
- Reads `?tenant=` parameter from current URL
- Extracts tenant identifier

#### Automatic Parameter Preservation
```javascript
const tenantAwarePath = getTenantAwareHref('/admin/dashboard/properties');
// If tenant=school-a exists: /admin/dashboard/properties?tenant=school-a
// If no tenant: /admin/dashboard/properties
```

#### Theme Loading
```javascript
// On page load and tab visibility change
if (tenant) {
  // Fetch tenant config from API
  // Apply theme color to CSS variables
  document.documentElement.style.setProperty('--primary-color', color);
  sessionStorage.setItem('tenant_theme_color', color);
  document.title = serverName;
} else {
  // Reset to default
  document.documentElement.style.setProperty('--primary-color', '#0056D2');
  sessionStorage.removeItem('tenant_theme_color');
}
```

### 3. Event Listeners

#### Visibility Change (Tab Switching)
```javascript
document.addEventListener('visibilitychange', () => {
  if (!document.hidden) {
    loadTenantTheme(); // Reload theme when tab becomes visible
  }
});
```

#### URL Change (Browser Navigation)
```javascript
window.addEventListener('popstate', () => {
  loadTenantTheme(); // Reload theme on back/forward navigation
});
```

### 4. Storage Strategy

**sessionStorage** (Tab-specific):
- Each browser tab maintains its own theme color
- No cross-tab interference
- Automatically cleared when tab closes

**Why not localStorage?**
- localStorage is shared across all tabs
- Would cause theme colors to interfere between tabs
- sessionStorage provides tab isolation

## Usage Examples

### In Sidebar Navigation
```typescript
import { getTenantAwareHref } from '../../utils/tenantNavigation';

const handleNavigation = (itemId: string) => {
  const path = `/admin/dashboard/${itemId}`;
  const fullPath = getTenantAwareHref(path); // Automatically adds tenant param
  navigate(fullPath);
};
```

### In Any Component
```typescript
import { getTenantAwareHref, getCurrentTenant } from '../../utils/tenantNavigation';

// Get current tenant
const tenant = getCurrentTenant(); // Returns 'school-a' or null

// Navigate with tenant
const link = getTenantAwareHref('/admin/dashboard/users');
// Result: /admin/dashboard/users?tenant=school-a
```

## Flow Diagram

```
User Opens Tenant
    ↓
URL: /admin/dashboard?tenant=school-a
    ↓
AdminDashboard loads → Reads tenant param → Fetches config → Applies theme
    ↓
User Clicks "Properties" in Sidebar
    ↓
getTenantAwareHref('/admin/dashboard/properties')
    ↓
Navigates to: /admin/dashboard/properties?tenant=school-a
    ↓
Theme reloads → Applies school-a theme
    ↓
User Switches to Another Tab (different tenant)
    ↓
visibilitychange event → Reloads theme for current tab's tenant
```

## Benefits

1. **Automatic**: No manual tenant parameter management needed
2. **Consistent**: Tenant context preserved across all navigation
3. **Isolated**: Each tab maintains its own tenant theme
4. **Dynamic**: Theme updates when switching tabs or navigating
5. **Clean**: Single utility function for all navigation
6. **Maintainable**: Change tenant logic in one place

## API Endpoints Used

```
GET /api/admin/configuration/tenants
- Returns list of all tenants

GET /api/admin/configuration/tenants/{tenant_id}
- Returns specific tenant configuration including:
  - theme_color
  - server_name
  - currency
  - language
  - feature flags
```

## Configuration Example

```javascript
// Tenant Config Response
{
  "id": "tenant-123",
  "name": "School A",
  "domain": "school-a.example.com",
  "configs": {
    "theme_color": "#FF5733",
    "server_name": "School A Server",
    "currency": "INR",
    "language": "en",
    "max_properties": 100,
    "enable_analytics": true
  }
}
```

## Testing

### Test Scenario 1: Single Tenant Navigation
1. Open: `http://localhost:5173/admin/dashboard?tenant=school-a`
2. Click "Properties" → URL becomes `/admin/dashboard/properties?tenant=school-a`
3. Click "Users" → URL becomes `/admin/dashboard/users?tenant=school-a`
4. Theme color remains consistent (school-a's color)

### Test Scenario 2: Multi-Tab Isolation
1. Tab 1: Open `?tenant=school-a` (blue theme)
2. Tab 2: Open `?tenant=school-b` (red theme)
3. Switch between tabs → Each maintains its own theme color
4. No cross-tab interference

### Test Scenario 3: No Tenant (Default)
1. Open: `http://localhost:5173/admin/dashboard`
2. Navigate to any page → No tenant parameter added
3. Theme color: Default #0056D2 (blue)

## Troubleshooting

### Theme not updating when switching tabs?
- Check if `visibilitychange` event listener is attached
- Verify sessionStorage has `tenant_theme_color` key
- Check browser console for API errors

### Tenant parameter lost during navigation?
- Ensure `getTenantAwareHref()` is used for all navigation
- Check if navigate() is called with the returned path
- Verify URL has tenant parameter before navigation

### Theme persisting across tabs?
- Confirm using sessionStorage (not localStorage)
- Check if each tab has separate sessionStorage
- Clear browser cache and test again

## Future Enhancements

1. **Tenant Switching UI**: Add dropdown to switch tenants without URL change
2. **Tenant Presets**: Save favorite tenant configurations
3. **Multi-Tenant Dashboard**: View multiple tenants side-by-side
4. **Tenant Analytics**: Track usage per tenant
5. **Tenant Permissions**: Role-based access per tenant
