# Quick FastAPI backend fix for registration endpoints
from fastapi import FastAPI, HTTPException
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel
import random
import string

app = FastAPI()

# Add CORS middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=["http://localhost:3000", "http://localhost:3001"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# In-memory storage (replace with database)
otp_storage = {}
users = {}

class EmailRequest(BaseModel):
    email: str

class OTPVerifyRequest(BaseModel):
    email: str
    otp_code: str
    password: str

@app.post("/api/auth/send-registration-otp")
async def send_registration_otp(request: EmailRequest):
    # Generate 6-digit OTP
    otp = ''.join(random.choices(string.digits, k=6))
    otp_storage[request.email] = otp
    
    print(f"OTP for {request.email}: {otp}")  # In production, send via email
    
    return {
        "success": True,
        "message": "OTP sent successfully",
        "otp": otp  # Remove this in production
    }

@app.post("/api/auth/verify-registration-otp")
async def verify_registration_otp(request: OTPVerifyRequest):
    stored_otp = otp_storage.get(request.email)
    
    if not stored_otp:
        raise HTTPException(status_code=400, detail="No OTP found for this email")
    
    if stored_otp != request.otp_code:
        raise HTTPException(status_code=400, detail="Invalid OTP")
    
    # Create user (replace with database logic)
    user_id = len(users) + 1
    users[request.email] = {
        "id": user_id,
        "email": request.email,
        "password": request.password  # Hash this in production
    }
    
    # Clean up OTP
    del otp_storage[request.email]
    
    return {
        "success": True,
        "message": "Registration successful",
        "access_token": f"fake_token_{user_id}",
        "user_id": user_id,
        "email": request.email
    }

if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8090)