import { PhysicalLevelTechnique } from "../metadata/tile/physicalLevelTechnique";
import { decodeComponentwiseDeltaVec2, decodeComponentwiseDeltaVec2Scaled, decodeDeltaRleInt32, decodeDeltaRleInt64, decodeFastPfor, decodeUnsignedConstRleInt32, decodeUnsignedConstRleInt64, decodeUnsignedRleInt32, decodeUnsignedRleInt64, decodeUnsignedRleFloat64, decodeVarintInt32, decodeVarintInt64, decodeVarintFloat64, decodeZigZagInt32, decodeZigZagInt64, decodeZigZagFloat64, decodeZigZagConstRleInt32, decodeZigZagConstRleInt64, decodeZigZagDeltaInt32, decodeZigZagDeltaInt64, decodeZigZagDeltaFloat64, decodeZigZagSequenceRleInt32, decodeZigZagSequenceRleInt64, decodeZigZagInt32Value, decodeZigZagInt64Value, fastInverseDelta, inverseDelta, decodeRleDeltaInt32, decodeZigZagDeltaOfDeltaInt32, decodeZigZagRleDeltaInt32, decodeZigZagRleInt32, decodeZigZagRleInt64, decodeZigZagRleFloat64, } from "./integerDecodingUtils";
import { LogicalLevelTechnique } from "../metadata/tile/logicalLevelTechnique";
import BitVector from "../vector/flat/bitVector";
import { VectorType } from "../vector/vectorType";
import { unpackNullable } from "./unpackNullableUtils";
export function decodeIntStream(data, offset, streamMetadata, isSigned, scalingData, nullabilityBuffer) {
    const values = decodePhysicalLevelTechnique(data, offset, streamMetadata);
    return decodeInt32(values, streamMetadata, isSigned, scalingData, nullabilityBuffer);
}
export function decodeLengthStreamToOffsetBuffer(data, offset, streamMetadata) {
    const values = decodePhysicalLevelTechnique(data, offset, streamMetadata);
    return decodeLengthToOffsetBuffer(values, streamMetadata);
}
function decodePhysicalLevelTechnique(data, offset, streamMetadata) {
    const physicalLevelTechnique = streamMetadata.physicalLevelTechnique;
    if (physicalLevelTechnique === PhysicalLevelTechnique.FAST_PFOR) {
        return decodeFastPfor(data, streamMetadata.numValues, streamMetadata.byteLength, offset);
    }
    if (physicalLevelTechnique === PhysicalLevelTechnique.VARINT) {
        return decodeVarintInt32(data, offset, streamMetadata.numValues);
    }
    if (physicalLevelTechnique === PhysicalLevelTechnique.NONE) {
        const dataOffset = offset.get();
        const byteLength = streamMetadata.byteLength;
        offset.add(byteLength);
        //TODO: use Byte Rle for geometry type encoding
        const slice = data.subarray(dataOffset, offset.get());
        return new Int32Array(slice);
    }
    throw new Error("Specified physicalLevelTechnique is not supported (yet).");
}
export function decodeConstIntStream(data, offset, streamMetadata, isSigned) {
    const values = decodePhysicalLevelTechnique(data, offset, streamMetadata);
    if (values.length === 1) {
        const value = values[0];
        return isSigned ? decodeZigZagInt32Value(value) : value;
    }
    return isSigned ? decodeZigZagConstRleInt32(values) : decodeUnsignedConstRleInt32(values);
}
export function decodeSequenceIntStream(data, offset, streamMetadata) {
    const values = decodePhysicalLevelTechnique(data, offset, streamMetadata);
    return decodeZigZagSequenceRleInt32(values);
}
export function decodeSequenceLongStream(data, offset, streamMetadata) {
    const values = decodeVarintInt64(data, offset, streamMetadata.numValues);
    return decodeZigZagSequenceRleInt64(values);
}
export function decodeLongStream(data, offset, streamMetadata, isSigned, nullabilityBuffer) {
    const values = decodeVarintInt64(data, offset, streamMetadata.numValues);
    return decodeInt64(values, streamMetadata, isSigned, nullabilityBuffer);
}
export function decodeLongFloat64Stream(data, offset, streamMetadata, isSigned) {
    const values = decodeVarintFloat64(data, offset, streamMetadata.numValues);
    return decodeFloat64(values, streamMetadata, isSigned);
}
export function decodeConstLongStream(data, offset, streamMetadata, isSigned) {
    const values = decodeVarintInt64(data, offset, streamMetadata.numValues);
    if (values.length === 1) {
        const value = values[0];
        return isSigned ? decodeZigZagInt64Value(value) : value;
    }
    return isSigned ? decodeZigZagConstRleInt64(values) : decodeUnsignedConstRleInt64(values);
}
/**
 * This method decodes integer streams.
 * Currently the encoder uses only fixed combinations of encodings.
 * For performance reasons it is also uses a fixed combination of the encodings on the decoding side.
 * The following encodings and combinations are used:
 *   - Morton Delta -> always sorted so not ZigZag encoding needed
 *   - Delta -> currently always in combination with ZigZag encoding
 *   - Rle -> in combination with ZigZag encoding if data type is signed
 *   - Delta Rle
 *   - Componentwise Delta -> always ZigZag encoding is used
 */
function decodeInt32(values, streamMetadata, isSigned, scalingData, nullabilityBuffer) {
    switch (streamMetadata.logicalLevelTechnique1) {
        case LogicalLevelTechnique.DELTA:
            if (streamMetadata.logicalLevelTechnique2 === LogicalLevelTechnique.RLE) {
                const rleMetadata = streamMetadata;
                if (!nullabilityBuffer) {
                    return decodeDeltaRleInt32(values, rleMetadata.runs, rleMetadata.numRleValues);
                }
                values = decodeUnsignedRleInt32(values, rleMetadata.runs, rleMetadata.numRleValues);
            }
            decodeZigZagDeltaInt32(values);
            break;
        case LogicalLevelTechnique.RLE:
            values = decodeRleInt32(values, streamMetadata, isSigned);
            break;
        case LogicalLevelTechnique.MORTON:
            fastInverseDelta(values);
            break;
        case LogicalLevelTechnique.COMPONENTWISE_DELTA:
            if (scalingData && !nullabilityBuffer) {
                decodeComponentwiseDeltaVec2Scaled(values, scalingData.scale, scalingData.min, scalingData.max);
                return values;
            }
            decodeComponentwiseDeltaVec2(values);
            break;
        case LogicalLevelTechnique.NONE:
            if (isSigned) {
                decodeZigZagInt32(values);
            }
            break;
        default:
            throw new Error(`The specified Logical level technique is not supported: ${streamMetadata.logicalLevelTechnique1}`);
    }
    if (nullabilityBuffer) {
        return unpackNullable(values, nullabilityBuffer, 0);
    }
    return values;
}
function decodeInt64(values, streamMetadata, isSigned, nullabilityBuffer) {
    switch (streamMetadata.logicalLevelTechnique1) {
        case LogicalLevelTechnique.DELTA:
            if (streamMetadata.logicalLevelTechnique2 === LogicalLevelTechnique.RLE) {
                const rleMetadata = streamMetadata;
                if (!nullabilityBuffer) {
                    return decodeDeltaRleInt64(values, rleMetadata.runs, rleMetadata.numRleValues);
                }
                values = decodeUnsignedRleInt64(values, rleMetadata.runs, rleMetadata.numRleValues);
            }
            decodeZigZagDeltaInt64(values);
            break;
        case LogicalLevelTechnique.RLE:
            values = decodeRleInt64(values, streamMetadata, isSigned);
            break;
        case LogicalLevelTechnique.NONE:
            if (isSigned) {
                decodeZigZagInt64(values);
            }
            break;
        default:
            throw new Error(`The specified Logical level technique is not supported: ${streamMetadata.logicalLevelTechnique1}`);
    }
    if (nullabilityBuffer) {
        return unpackNullable(values, nullabilityBuffer, 0n);
    }
    return values;
}
export function decodeFloat64(values, streamMetadata, isSigned) {
    switch (streamMetadata.logicalLevelTechnique1) {
        case LogicalLevelTechnique.DELTA:
            if (streamMetadata.logicalLevelTechnique2 === LogicalLevelTechnique.RLE) {
                const rleMetadata = streamMetadata;
                values = decodeUnsignedRleFloat64(values, rleMetadata.runs, rleMetadata.numRleValues);
            }
            decodeZigZagDeltaFloat64(values);
            return values;
        case LogicalLevelTechnique.RLE:
            return decodeRleFloat64(values, streamMetadata, isSigned);
        case LogicalLevelTechnique.NONE:
            if (isSigned) {
                decodeZigZagFloat64(values);
            }
            return values;
        default:
            throw new Error(`The specified Logical level technique is not supported: ${streamMetadata.logicalLevelTechnique1}`);
    }
}
function decodeLengthToOffsetBuffer(values, streamMetadata) {
    if (streamMetadata.logicalLevelTechnique1 === LogicalLevelTechnique.DELTA &&
        streamMetadata.logicalLevelTechnique2 === LogicalLevelTechnique.NONE) {
        return decodeZigZagDeltaOfDeltaInt32(values);
    }
    if (streamMetadata.logicalLevelTechnique1 === LogicalLevelTechnique.RLE &&
        streamMetadata.logicalLevelTechnique2 === LogicalLevelTechnique.NONE) {
        const rleMetadata = streamMetadata;
        return decodeRleDeltaInt32(values, rleMetadata.runs, rleMetadata.numRleValues);
    }
    if (streamMetadata.logicalLevelTechnique1 === LogicalLevelTechnique.NONE &&
        streamMetadata.logicalLevelTechnique2 === LogicalLevelTechnique.NONE) {
        //TODO: use fastInverseDelta again and check what are the performance problems in zoom 14
        //fastInverseDelta(values);
        inverseDelta(values);
        const offsets = new Uint32Array(streamMetadata.numValues + 1);
        offsets[0] = 0;
        offsets.set(values, 1);
        return offsets;
    }
    if (streamMetadata.logicalLevelTechnique1 === LogicalLevelTechnique.DELTA &&
        streamMetadata.logicalLevelTechnique2 === LogicalLevelTechnique.RLE) {
        const rleMetadata = streamMetadata;
        const decodedValues = decodeZigZagRleDeltaInt32(values, rleMetadata.runs, rleMetadata.numRleValues);
        fastInverseDelta(decodedValues);
        return decodedValues;
    }
    throw new Error("Only delta encoding is supported for transforming length to offset streams yet.");
}
export function getVectorType(streamMetadata, sizeOrNullabilityBuffer, data, offset) {
    const logicalLevelTechnique1 = streamMetadata.logicalLevelTechnique1;
    if (logicalLevelTechnique1 === LogicalLevelTechnique.RLE) {
        return streamMetadata.runs === 1 ? VectorType.CONST : VectorType.FLAT;
    }
    if (logicalLevelTechnique1 !== LogicalLevelTechnique.DELTA ||
        streamMetadata.logicalLevelTechnique2 !== LogicalLevelTechnique.RLE) {
        return streamMetadata.numValues === 1 ? VectorType.CONST : VectorType.FLAT;
    }
    const numFeatures = sizeOrNullabilityBuffer instanceof BitVector ? sizeOrNullabilityBuffer.size() : sizeOrNullabilityBuffer;
    const rleMetadata = streamMetadata;
    if (rleMetadata.numRleValues !== numFeatures) {
        return VectorType.FLAT;
    }
    // Single run is always a sequence
    if (rleMetadata.runs === 1) {
        return VectorType.SEQUENCE;
    }
    if (rleMetadata.runs !== 2) {
        return streamMetadata.numValues === 1 ? VectorType.CONST : VectorType.FLAT;
    }
    // Two runs can be a sequence if both deltas are equal to 1
    const savedOffset = offset.get();
    let values;
    if (streamMetadata.physicalLevelTechnique === PhysicalLevelTechnique.VARINT) {
        values = decodeVarintInt32(data, offset, 4);
    }
    else {
        const byteOffset = offset.get();
        values = new Int32Array(data.buffer, data.byteOffset + byteOffset, 4);
    }
    offset.set(savedOffset);
    // Check if both deltas are encoded 1
    const zigZagOne = 2;
    if (values[2] === zigZagOne && values[3] === zigZagOne) {
        return VectorType.SEQUENCE;
    }
    return streamMetadata.numValues === 1 ? VectorType.CONST : VectorType.FLAT;
}
function decodeRleInt32(data, streamMetadata, isSigned) {
    return isSigned
        ? decodeZigZagRleInt32(data, streamMetadata.runs, streamMetadata.numRleValues)
        : decodeUnsignedRleInt32(data, streamMetadata.runs, streamMetadata.numRleValues);
}
function decodeRleInt64(data, streamMetadata, isSigned) {
    return isSigned
        ? decodeZigZagRleInt64(data, streamMetadata.runs, streamMetadata.numRleValues)
        : decodeUnsignedRleInt64(data, streamMetadata.runs, streamMetadata.numRleValues);
}
function decodeRleFloat64(data, streamMetadata, isSigned) {
    return isSigned
        ? decodeZigZagRleFloat64(data, streamMetadata.runs, streamMetadata.numRleValues)
        : decodeUnsignedRleFloat64(data, streamMetadata.runs, streamMetadata.numRleValues);
}
//# sourceMappingURL=integerStreamDecoder.js.map