import { ScalarType } from "../metadata/tileset/tilesetMetadata";
import BitVector from "../vector/flat/bitVector";
import { decodeStreamMetadata } from "../metadata/tile/streamMetadataDecoder";
import { VectorType } from "../vector/vectorType";
import { BooleanFlatVector } from "../vector/flat/booleanFlatVector";
import { DoubleFlatVector } from "../vector/flat/doubleFlatVector";
import { FloatFlatVector } from "../vector/flat/floatFlatVector";
import { LongConstVector } from "../vector/constant/longConstVector";
import { LongFlatVector } from "../vector/flat/longFlatVector";
import { IntFlatVector } from "../vector/flat/intFlatVector";
import { IntConstVector } from "../vector/constant/intConstVector";
import { decodeBooleanRle, decodeDoublesLE, decodeFloatsLE, skipColumn } from "./decodingUtils";
import { decodeConstIntStream, decodeConstLongStream, decodeIntStream, decodeLongStream, decodeSequenceIntStream, decodeSequenceLongStream, getVectorType, } from "./integerStreamDecoder";
import { IntSequenceVector } from "../vector/sequence/intSequenceVector";
import { LongSequenceVector } from "../vector/sequence/longSequenceVector";
import { decodeSharedDictionary, decodeString } from "./stringDecoder";
export function decodePropertyColumn(data, offset, columnMetadata, numStreams, numFeatures, propertyColumnNames) {
    if (columnMetadata.type === "scalarType") {
        if (propertyColumnNames && !propertyColumnNames.has(columnMetadata.name)) {
            skipColumn(numStreams, data, offset);
            return null;
        }
        return decodeScalarPropertyColumn(numStreams, data, offset, numFeatures, columnMetadata.scalarType, columnMetadata);
    }
    if (numStreams === 0) {
        return null;
    }
    return decodeSharedDictionary(data, offset, columnMetadata, numFeatures, propertyColumnNames);
}
function decodeScalarPropertyColumn(numStreams, data, offset, numFeatures, column, columnMetadata) {
    let nullabilityBuffer = null;
    let numValues = 0;
    if (numStreams === 0) {
        /* Skip since this column has no values */
        return null;
    }
    // Read nullability stream if column is nullable
    if (columnMetadata.nullable) {
        const presentStreamMetadata = decodeStreamMetadata(data, offset);
        numValues = presentStreamMetadata.numValues;
        const streamDataStart = offset.get();
        const presentVector = decodeBooleanRle(data, numValues, presentStreamMetadata.byteLength, offset);
        offset.set(streamDataStart + presentStreamMetadata.byteLength);
        nullabilityBuffer = new BitVector(presentVector, presentStreamMetadata.numValues);
    }
    const sizeOrNullabilityBuffer = nullabilityBuffer ?? numFeatures;
    const scalarType = column.physicalType;
    switch (scalarType) {
        case ScalarType.UINT_32:
        case ScalarType.INT_32:
            return decodeIntColumn(data, offset, columnMetadata, column, sizeOrNullabilityBuffer);
        case ScalarType.STRING:
            // In embedded format: numStreams includes nullability stream if column is nullable
            const stringDataStreams = columnMetadata.nullable ? numStreams - 1 : numStreams;
            return decodeString(columnMetadata.name, data, offset, stringDataStreams, nullabilityBuffer);
        case ScalarType.BOOLEAN:
            return decodeBooleanColumn(data, offset, columnMetadata, numFeatures, sizeOrNullabilityBuffer);
        case ScalarType.UINT_64:
        case ScalarType.INT_64:
            return decodeLongColumn(data, offset, columnMetadata, sizeOrNullabilityBuffer, column);
        case ScalarType.FLOAT:
            return decodeFloatColumn(data, offset, columnMetadata, sizeOrNullabilityBuffer);
        case ScalarType.DOUBLE:
            return decodeDoubleColumn(data, offset, columnMetadata, sizeOrNullabilityBuffer);
        default:
            throw new Error(`The specified data type for the field is currently not supported: ${column}`);
    }
}
function decodeBooleanColumn(data, offset, column, numFeatures, sizeOrNullabilityBuffer) {
    const dataStreamMetadata = decodeStreamMetadata(data, offset);
    const numValues = dataStreamMetadata.numValues;
    const streamDataStart = offset.get();
    const nullabilityBuffer = isNullabilityBuffer(sizeOrNullabilityBuffer) ? sizeOrNullabilityBuffer : undefined;
    const dataStream = decodeBooleanRle(data, numValues, dataStreamMetadata.byteLength, offset, nullabilityBuffer);
    offset.set(streamDataStart + dataStreamMetadata.byteLength);
    const dataVector = new BitVector(dataStream, numValues);
    return new BooleanFlatVector(column.name, dataVector, sizeOrNullabilityBuffer);
}
function decodeFloatColumn(data, offset, column, sizeOrNullabilityBuffer) {
    const dataStreamMetadata = decodeStreamMetadata(data, offset);
    const nullabilityBuffer = isNullabilityBuffer(sizeOrNullabilityBuffer) ? sizeOrNullabilityBuffer : undefined;
    const dataStream = decodeFloatsLE(data, offset, dataStreamMetadata.numValues, nullabilityBuffer);
    return new FloatFlatVector(column.name, dataStream, sizeOrNullabilityBuffer);
}
function decodeDoubleColumn(data, offset, column, sizeOrNullabilityBuffer) {
    const dataStreamMetadata = decodeStreamMetadata(data, offset);
    const nullabilityBuffer = isNullabilityBuffer(sizeOrNullabilityBuffer) ? sizeOrNullabilityBuffer : undefined;
    const dataStream = decodeDoublesLE(data, offset, dataStreamMetadata.numValues, nullabilityBuffer);
    return new DoubleFlatVector(column.name, dataStream, sizeOrNullabilityBuffer);
}
function decodeLongColumn(data, offset, column, sizeOrNullabilityBuffer, scalarColumn) {
    const dataStreamMetadata = decodeStreamMetadata(data, offset);
    const vectorType = getVectorType(dataStreamMetadata, sizeOrNullabilityBuffer, data, offset);
    const isSigned = scalarColumn.physicalType === ScalarType.INT_64;
    if (vectorType === VectorType.FLAT) {
        const nullabilityBuffer = isNullabilityBuffer(sizeOrNullabilityBuffer) ? sizeOrNullabilityBuffer : undefined;
        const dataStream = decodeLongStream(data, offset, dataStreamMetadata, isSigned, nullabilityBuffer);
        return new LongFlatVector(column.name, dataStream, sizeOrNullabilityBuffer);
    }
    else if (vectorType === VectorType.SEQUENCE) {
        const id = decodeSequenceLongStream(data, offset, dataStreamMetadata);
        return new LongSequenceVector(column.name, id[0], id[1], dataStreamMetadata.numRleValues);
    }
    else {
        const constValue = decodeConstLongStream(data, offset, dataStreamMetadata, isSigned);
        return new LongConstVector(column.name, constValue, sizeOrNullabilityBuffer);
    }
}
function decodeIntColumn(data, offset, column, scalarColumn, sizeOrNullabilityBuffer) {
    const dataStreamMetadata = decodeStreamMetadata(data, offset);
    const vectorType = getVectorType(dataStreamMetadata, sizeOrNullabilityBuffer, data, offset);
    const isSigned = scalarColumn.physicalType === ScalarType.INT_32;
    if (vectorType === VectorType.FLAT) {
        const nullabilityBuffer = isNullabilityBuffer(sizeOrNullabilityBuffer) ? sizeOrNullabilityBuffer : undefined;
        const dataStream = decodeIntStream(data, offset, dataStreamMetadata, isSigned, undefined, nullabilityBuffer);
        return new IntFlatVector(column.name, dataStream, sizeOrNullabilityBuffer);
    }
    else if (vectorType === VectorType.SEQUENCE) {
        const id = decodeSequenceIntStream(data, offset, dataStreamMetadata);
        return new IntSequenceVector(column.name, id[0], id[1], dataStreamMetadata.numRleValues);
    }
    else {
        const constValue = decodeConstIntStream(data, offset, dataStreamMetadata, isSigned);
        return new IntConstVector(column.name, constValue, sizeOrNullabilityBuffer);
    }
}
function isNullabilityBuffer(sizeOrNullabilityBuffer) {
    return sizeOrNullabilityBuffer instanceof BitVector;
}
//# sourceMappingURL=propertyDecoder.js.map