
import typescript from '@rollup/plugin-typescript';
import resolve from '@rollup/plugin-node-resolve';
import replace from '@rollup/plugin-replace';
import commonjs from '@rollup/plugin-commonjs';
import terser from '@rollup/plugin-terser';
import strip from '@rollup/plugin-strip';
import {type Plugin} from 'rollup';
import json from '@rollup/plugin-json';
import {visualizer} from 'rollup-plugin-visualizer';

const {BUNDLE} = process.env;
const stats = BUNDLE === 'stats';

// Common set of plugins/transformations shared across different rollup
// builds (main maplibre bundle, style-spec package, benchmarks bundle)

export const nodeResolve = resolve({
    browser: true,
    preferBuiltins: false
});

export const plugins = (production: boolean): Plugin[] => [
    json(),
    // https://github.com/zaach/jison/issues/351
    replace({
        preventAssignment: true,
        include: /\/jsonlint-lines-primitives\/lib\/jsonlint.js/,
        delimiters: ['', ''],
        values: {
            '_token_stack:': ''
        }
    }),
    production && strip({
        sourceMap: true,
        functions: ['PerformanceUtils.*']
    }),
    production && terser({
        compress: {
            pure_getters: true,
            passes: 3
        },
        sourceMap: true
    }),
    nodeResolve,
    typescript(),
    commonjs({
        // global keyword handling causes Webpack compatibility issues, so we disabled it:
        // https://github.com/mapbox/mapbox-gl-js/pull/6956
        ignoreGlobal: true
    }),
    // generate bundle stats in multiple formats (treemap, sunburst, etc...)
    ...(stats ? (['treemap', 'sunburst', 'flamegraph', 'network'] as const).map(template =>
        visualizer({
            template: template,
            title: `gl-js-${template}`,
            filename: `staging/${template}.html`,
            gzipSize: true,
            brotliSize: true,
            sourcemap: true,
            open: true
        })
    ) : [])
].filter(Boolean) as Plugin[];

export const watchStagingPlugin: Plugin = {
    name: 'watch-external',
    buildStart() {
        this.addWatchFile('staging/maplibregl/index.js');
        this.addWatchFile('staging/maplibregl/shared.js');
        this.addWatchFile('staging/maplibregl/worker.js');
    }
};
