import React, { useState, useEffect, Suspense, useRef } from "react";
import {
  BrowserRouter as Router,
  Routes,
  Route,
  useLocation,
} from "react-router-dom";
import { AnimatePresence } from "framer-motion";
import PageTransition from "./components/ui/PageTransition";
import { ImpersonationBanner } from "./components/admin-dashboard/ImpersonationBanner";
import { Header } from "./components/Header";
import { HeroSection } from "./components/HeroSection";
import { FeaturedProperties } from "./components/FeaturedProperties";
import { BiddingProperties } from "./components/BiddingProperties";
import { FinancialTools } from "./components/FinancialTools";
import { ImmersiveExperience } from "./components/ImmersiveExperience";
import { NALServices } from "./components/NALServices";
import { KnowledgeHub } from "./components/KnowledgeHub";
import { SearchByCities } from "./components/SearchByCities";
import { HeatmapPage } from "./pages/HeatmapPage";
import { TopPerformingMarkets } from "./pages/TopPerformingMarkets";
import { MicroMarketsDashboard } from "./pages/MicroMarketsDashboard";
import { RatesAndTrends } from "./pages/RatesAndTrends";
import { SmartLandMapsPage } from "./pages/SmartLandMapsPage";
import { MicroMarketPerformanceHub } from "./pages/MicroMarketPerformanceHub";
import { CampaignManagement } from "./components/CampaignManagement";
import { Footer } from "./components/Footer";
import { PropertyDetail } from "./components/PropertyDetail";
import { PropertyVisitBooking } from "./components/PropertyVisitBooking";
import { ContactAgentForm } from "./components/ContactAgentForm";
import BiddingPropertyDetail from "./pages/BiddingPropertyDetail";
import { Properties } from "./components/Properties";
import { AgentAuth } from "./components/AgentAuth";
import { BuyerAuth } from "./components/BuyerAuth";
import AgentDashboard from "./components/AgentDashboard";
import AgentProfileCompletion from "./components/agent-dashboard/AgentProfileCompletion";
import AgentProfileGuard from "./components/agent-dashboard/AgentProfileGuard";
import SellerProfileCompletion from "./components/SellerProfileCompletion";
import { SellerAuth } from "./components/SellerAuth";
import SellerDashboard from "./pages/SellerDashboard";
import { AdminLogin } from "./components/AdminLogin";
import { AdminDashboard } from "./pages/AdminDashboard";
import { AboutUs } from "./components/AboutUs";
import { ContactUs } from "./components/ContactUs";
import { ListingForm } from "./pages/ListingForm";
import { SinglePageListingForm } from "./pages/SinglePageListingForm";
import { ListPropertyPage } from "./pages/ListPropertyPage";
import { EMICalculator } from "./pages/EMICalculator";
import { LoanEligibility } from "./pages/LoanEligibility";
import BudgetCalculator from "./pages/BudgetCalculator";
import AreaConverter from "./pages/AreaConverter";
import { StampDutyCalculator } from "./pages/StampDutyCalculator";
import BlogDetail from "./pages/BlogDetail";
import { PremiumPackages } from "./pages/PremiumPackages";
import { BuyerDashboard } from "./pages/BuyerDashboard";
import { CheckInPage } from "./components/visits/CheckInPage";
import { VerifyBadge } from "./pages/VerifyBadge";
import { MyBadges } from "./pages/MyBadges";
import Profile from "./components/Profile/Profile";
import YourActivity from "./components/YourActivity";
import { ListingProvider } from "./context/ListingContext";
import { ScrollToTop } from "./components/ScrollToTop";
import Chatbot from "./components/chatbot.jsx";
import { Toaster } from "sonner";
import { ErrorBoundary } from "./components/ErrorBoundary";
import { SafeWrapper } from "./components/SafeWrapper";
import { safeLocalStorage, safeCall } from "./utils/safeAccess";
import ProfileCompletionModal from "./components/Profile/ProfileCompletionModal";
import { getProfile } from "./api/profile";
import { ProfileSkeleton } from "./components/ui/skeleton";
import TopLoader from "./components/ui/TopLoader";
import WhatsAppFloat from "./components/WhatsAppFloat";
import { startTokenRefreshChecker } from "./api/auth";
import {
  initializeActivityTracking,
  cleanupActivityTracking,
} from "./utils/auth";
import { AdminRoute } from "./components/AdminRoute";
import { UserRoute } from "./components/UserRoute";
// import "./utils/makeAdmin"; // Load admin utility - file not found
import AuctionDemo from "./components/AuctionDemo";
import AuctionRegistration from "./pages/AuctionRegistration";
import { DocumentVerification } from "./pages/DocumentVerification";
import DocumentVerificationResults from "./pages/DocumentVerificationResults";
import DocumentVerificationFinal from "./pages/DocumentVerificationFinal";
import RibilScorecardPage from "./pages/RibilScorecardPage";
import { LegalGuidelines } from "./pages/LegalGuidelines";
import { PrivacyPolicy } from "./pages/PrivacyPolicy";
import { HelpCenter } from "./pages/HelpCenter";
import { NALStories } from "./pages/NALStories";
import ARViewer from "./components/ARViewer";
import CampaignManagementPage from "./pages/CampaignManagement";
import { CampaignForm } from "./components/CampaignForm";
import { LoadingProvider, useLoading } from "./context/LoadingContext";
import LoadingDots from "./components/ui/LoadingDots";
import { CompareProvider } from "./context/CompareContext";
import { CompareBar } from "./components/CompareBar";
import { ComparePage } from "./pages/ComparePage";
import { MapSelector } from "./pages/MapSelector";
import { ServicePage } from "./pages/ServicePage";
import { AdvisoryPage } from "./pages/AdvisoryPage";
import { SolutionsPage } from "./pages/SolutionsPage";
import { DigitalVaultDemo } from "./pages/DigitalVaultDemo";
import { RiskCompliancePage } from "./pages/advisory/risk-compliance/RiskCompliancePage";
import { AdminRiskReviews } from "./pages/advisory/risk-compliance/AdminRiskReviews";
import MarketTrendsPage from "./pages/advisory/MarketTrendsPage";
import { RentalYieldPage } from "./pages/advisory/RentalYieldPage";
import { NRIAssistanceDashboard } from "./pages/advisory/NRIAssistanceClean";
import { NRIAssistance } from "./pages/advisory/NRIAssistancePublic";
import { CustomPropertyDossier } from "./pages/advisory/CustomPropertyDossier";
import SystemHealthDashboard from "./components/system-health-monitoring/components/SystemHealthDashboard";
import LegalHealthLanding from "./pages/advisory/LegalHealthLanding";
import CreateLegalHealthReport from "./pages/buyer/legal-health/CreateLegalHealthReport";
import MyLegalHealthReports from "./pages/buyer/legal-health/MyLegalHealthReports";
import ReportDetails from "./pages/buyer/legal-health/ReportDetails";
import UploadDocuments from "./pages/buyer/legal-health/UploadDocuments";

import { InvestmentInsights } from "./components/advisory/InvestmentInsights";
import { PricePrediction } from "./pages/price-prediction/PricePrediction";
import { TransactionHistory as TransactionHistoryPage } from "./pages/TransactionHistory";
import { TransactionDetails } from "./pages/TransactionDetails";
import PrivacySettings from "./components/PrivacySettings";
import { AdminEscrowList } from "./pages/AdminEscrowList";
import { AdminEscrowReview } from "./pages/AdminEscrowReview";
import { MyItineraries } from "./pages/itinerary/MyItineraries";
import { CreateItinerary } from "./pages/itinerary/CreateItinerary";
import { ItineraryDetail } from "./pages/itinerary/ItineraryDetail";
import { ItineraryDetail as ItineraryDetailPage } from "./pages/ItineraryDetail";
import { LiveTour } from "./pages/itinerary/LiveTour";
import { ValuationReportsPage } from "./pages/ValuationReportsPage";
import {
  ForgotPasswordPage,
  SellerForgotPasswordPage,
  AgentForgotPasswordPage,
} from "./pages/ForgotPasswordPage";
import { GoogleCallback } from "./pages/GoogleCallback";
import { OAuthCallback } from "./pages/OAuthCallback";
import ListingScoreDemo from "./components/ListingScoreDemo";
import AerialView from "./pages/AerialView";
import NegotiationPage from "./pages/NegotiationPage";
import AdminListingPackages from "./components/admin-dashboard/AdminListingPackages";
import OwnerListingPackages from "./components/buyer-dashboard/OwnerListingPackages";

// Import Promotions Components
import { AdminCoupons } from "./components/admin-dashboard/AdminCoupons";
import { AdminDiscountRules } from "./components/admin-dashboard/AdminDiscountRules";
import { AdminScholarships } from "./components/admin-dashboard/AdminScholarships";
import { UserPricingWithCoupons } from "./components/buyer-dashboard/UserPricingWithCoupons";
import { UserScholarships } from "./components/buyer-dashboard/UserScholarships";

// Import Support Ticketing
import SupportTickets from "./pages/SupportTickets";
import AdminSupport from "./pages/AdminSupport";

// Import Grievance Management
import GrievancePage from "./pages/GrievancePage";

// Import Entitlement Components
import UserEntitlementsDashboard from "./components/entitlements/UserEntitlementsDashboard";
import AdminEntitlementManager from "./components/entitlements/AdminEntitlementManager";

// Event Bus imports
import { EventBusProvider } from "./context/EventBusContext";
import { NotificationPanel } from "./components/NotificationPanel";
import { EventBusTest } from "./components/EventBusTest";

// Import KYC Forms
import { OwnerKYCForm } from "./components/owner-kyc/OwnerKYCForm";
import { AgentKYCForm } from "./components/agent-kyc/AgentKYCForm";
import { BuyerKYCForm } from "./components/buyer-kyc/BuyerKYCForm";
import KYCPage from "./pages/KYCPage";

// Import dedicated service pages
import {
  DigitalVault,
  DocumentVerification as DocumentVerificationService,
  ESignEStamp,
  HomeInsurance,
  LegalLoanAssistance,
  MarketAnalysis,
  PropertyManagement,
  PropertyValuation,
  RERACompliance,
  RERAGuidanceBuyers,
  RERARegistrationGuidance,
  TitleSearch,
  DigitalMarketingServices,
  // Premium service pages
  ESignEStampServices,
  PropertyValuationPremium,
  MarketAnalysisPremium,
  TitleSearchPremium,
  RERACompliancePremium,
  PropertyManagementPremium,
  HomeInsuranceAssistance,
  HomeBuyersGuidance,
  PropertyOwnersGuidance,
  FirstTimeBuyersGuidance,
  LegalLoanAssistancePremium,
} from "./pages/services";
import { EStampPage } from "./pages/services/EStampPage";
import { TitleSearchResults } from "./pages/TitleSearchResults";
import { TitleSearchInstantResults } from "./pages/TitleSearchInstantResults";
import { ContractManagement } from "./pages/services/ContractManagement";

const ProfileComplete = React.lazy(
  () => import("./components/Profile/ProfileComplete")
);

function HomePage() {
  const location = useLocation();
  const [showProfileModal, setShowProfileModal] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [isSellerOrAgent, setIsSellerOrAgent] = useState(false);

  // Check if user is seller or agent
  useEffect(() => {
    const agentToken = safeLocalStorage.getItem("agentToken");
    const sellerToken = safeLocalStorage.getItem("sellerToken");
    setIsSellerOrAgent(!!(agentToken || sellerToken));
  }, []);

  // Clear the admin homepage visit flag when component unmounts
  useEffect(() => {
    return () => {
      sessionStorage.removeItem("admin_visited_homepage");
    };
  }, []);

  useEffect(() => {
    // Don't show profile modal for admin users
    const accessToken = safeLocalStorage.getItem("access_token");
    if (accessToken) {
      try {
        const payload = JSON.parse(atob(accessToken.split(".")[1]));
        if (payload.is_admin === true) {
          return; // Skip profile modal for admin
        }
      } catch (e) { }
    }

    if (location.state?.showProfileModal) setShowProfileModal(true);
  }, [location.state]);

  useEffect(() => {
    let isMounted = true;

    safeCall(() => {
      const accessToken = safeLocalStorage.getItem("access_token");
      const agentToken = safeLocalStorage.getItem("agentToken");
      const sellerToken = safeLocalStorage.getItem("sellerToken");

      // Skip for agents/sellers
      if (agentToken || sellerToken) {
        setIsLoading(false);
        return;
      }

      // Don't show profile modal for admin users
      if (accessToken) {
        try {
          const payload = JSON.parse(atob(accessToken.split(".")[1]));
          if (payload.is_admin === true) {
            setIsLoading(false);
            return; // Skip profile modal for admin
          }
        } catch (e) { }
      }

      if (
        !accessToken ||
        safeLocalStorage.getItem("profile_modal_shown_this_login")
      ) {
        setIsLoading(false);
        return;
      }

      const profileCompleted =
        safeLocalStorage.getItem("profile_completed") === "true";
      if (profileCompleted) {
        setIsLoading(false);
        return;
      }

      const userStr = safeLocalStorage.getItem("user") || "{}";
      let user = {};
      try {
        user = JSON.parse(userStr);
      } catch (e) {
        console.warn("Failed to parse user data:", e);
      }

      if (user && (user as any).is_new_user) {
        setShowProfileModal(true);
        setIsLoading(false);
        return;
      }

      // Prevent multiple API calls
      if (sessionStorage.getItem("profile_check_in_progress")) {
        setIsLoading(false);
        return;
      }

      sessionStorage.setItem('profile_check_in_progress', 'true');

      getProfile()
        .then((data) => {
          if (!isMounted) return;

          if (data && data.profile_completion_percentage < 100) {
            setShowProfileModal(true);
          }
          setIsLoading(false);
          sessionStorage.removeItem("profile_check_in_progress");
        })
        .catch((err) => {
          if (!isMounted) return;

          console.error("Profile check failed:", err);
          setIsLoading(false);
          sessionStorage.removeItem("profile_check_in_progress");
        });
    });

    return () => {
      isMounted = false;
    };
  }, []); // Empty dependency array to run only once

  return (
    <>
      <TopLoader isLoading={isLoading} />
      <HeroSection />
      <FeaturedProperties />
      <BiddingProperties />
      <SearchByCities />
      {isSellerOrAgent && <CampaignManagement />}
      <FinancialTools />
      <ImmersiveExperience />
      <NALServices />
      <KnowledgeHub />
      <ProfileCompletionModal
        isOpen={showProfileModal}
        onClose={() => setShowProfileModal(false)}
        onComplete={() => setShowProfileModal(false)}
      />
    </>
  );
}

function AppContent() {
  const location = useLocation();
  const isDashboardRoute = location?.pathname?.includes("/dashboard") || false;
  const isAdminRoute = location?.pathname?.startsWith("/admin") || false;
  const isAuthRoute =
    location?.pathname === "/login" || location?.pathname === "/signup";

  // Get user ID for Event Bus
  const getUserId = () => {
    try {
      const accessToken = safeLocalStorage.getItem("access_token");
      if (accessToken) {
        const payload = JSON.parse(atob(accessToken.split(".")[1]));
        return payload.user_id || payload.sub || "anonymous";
      }
      return "anonymous";
    } catch {
      return "anonymous";
    }
  };

  const userId = getUserId();

  // Safe loading context access
  let isLoginLoading = false;
  let setLoginLoading: (value: boolean) => void = () => { };

  try {
    const loadingContext = useLoading();
    isLoginLoading = loadingContext?.isLoginLoading || false;
    setLoginLoading = loadingContext?.setLoginLoading || (() => { });
  } catch (error) {
    console.warn("Loading context not available:", error);
  }

  const handleLoginComplete = (loading: boolean) => {
    try {
      setLoginLoading(false);
      // Check if user is agent before redirecting
      const isAgentUser =
        safeLocalStorage.getItem("agentToken") ||
        safeLocalStorage.getItem("agentEmail");
      if (isAgentUser) {
        window.location.href = "/";
      } else {
        setLoginLoading(loading);
        if (!loading) {
          window.location.href = "/login";
        }
      }
    } catch (error) {
      console.error("Error in handleLoginComplete:", error);
    }
  };

  return (
    <>
      <SafeWrapper componentName="LoadingDots">
        <LoadingDots isLoading={isLoginLoading} />
      </SafeWrapper>
      <div className="min-h-screen bg-white flex flex-col">
        <ImpersonationBanner />
        {!isDashboardRoute && !isAdminRoute && !isAuthRoute && (
          <SafeWrapper componentName="Header">
            <Header />
            {/* Add Notification Panel to Header - only for logged-in users */}
            {userId !== "anonymous" && userId && (
              <div className="fixed top-4 right-4 z-50">
                <NotificationPanel />
              </div>
            )}
          </SafeWrapper>
        )}
        <main className="flex-1">
          <AnimatePresence mode="wait">
            <Routes location={location} key={location?.pathname || "default"}>
              {/* System Health Route */}
              <Route
                path="/system-health"
                element={
                  <PageTransition>
                    <SystemHealthDashboard />
                  </PageTransition>
                }
              />

              {/* Agent Routes */}
              <Route
                path="/agent"
                element={
                  <PageTransition>
                    <AgentAuth />
                  </PageTransition>
                }
              />
              <Route
                path="/agent/register"
                element={
                  <PageTransition>
                    <AgentAuth />
                  </PageTransition>
                }
              />
              <Route
                path="/agent/profile-completion"
                element={
                  <AgentProfileGuard>
                    <AgentProfileCompletion />
                  </AgentProfileGuard>
                }
              />
              <Route path="/agent/dashboard/*" element={<AgentDashboard />} />

              {/* Seller Routes */}
              <Route
                path="/seller"
                element={
                  <PageTransition>
                    <SellerAuth />
                  </PageTransition>
                }
              />
              <Route
                path="/seller/register"
                element={
                  <PageTransition>
                    <SellerAuth />
                  </PageTransition>
                }
              />
              <Route
                path="/seller/profile-completion"
                element={<SellerProfileCompletion />}
              />
              <Route
                path="/seller/dashboard/*"
                element={
                  <PageTransition>
                    <SellerDashboard />
                  </PageTransition>
                }
              />

              {/* Admin Routes (Protected) */}
              <Route
                path="/admin/login"
                element={
                  <PageTransition>
                    <AdminLogin />
                  </PageTransition>
                }
              />
              <Route
                path="/admin"
                element={
                  <PageTransition>
                    <AdminRoute>
                      <AdminDashboard />
                    </AdminRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/admin/dashboard/*"
                element={
                  <PageTransition>
                    <AdminRoute>
                      <AdminDashboard />
                    </AdminRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/admin/risk-reviews"
                element={
                  <PageTransition>
                    <AdminRoute>
                      <AdminRiskReviews />
                    </AdminRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/admin/escrows"
                element={
                  <PageTransition>
                    <AdminRoute>
                      <AdminEscrowList />
                    </AdminRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/admin/escrow/:id"
                element={
                  <PageTransition>
                    <AdminRoute>
                      <AdminEscrowReview />
                    </AdminRoute>
                  </PageTransition>
                }
              />

              {/* User Dashboard Route */}
              <Route
                path="/user/dashboard"
                element={
                  <PageTransition>
                    <UserRoute>
                      <BuyerDashboard />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/user/badges"
                element={
                  <PageTransition>
                    <UserRoute>
                      <MyBadges />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/check-in/:visitId"
                element={
                  <PageTransition>
                    <UserRoute>
                      <CheckInPage />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/verify-badge"
                element={
                  <PageTransition>
                    <VerifyBadge />
                  </PageTransition>
                }
              />

              {/* Itinerary Routes */}
              <Route
                path="/my-itineraries"
                element={
                  <PageTransition>
                    <UserRoute>
                      <MyItineraries />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/create-itinerary"
                element={
                  <PageTransition>
                    <UserRoute>
                      <CreateItinerary />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/itinerary/:id"
                element={
                  <PageTransition>
                    <UserRoute>
                      <ItineraryDetail />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/itinerary-detail/:id"
                element={
                  <PageTransition>
                    <UserRoute>
                      <ItineraryDetailPage />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/itinerary/:id/tour"
                element={
                  <PageTransition>
                    <UserRoute>
                      <LiveTour />
                    </UserRoute>
                  </PageTransition>
                }
              />

              {/* Common Pages */}
              <Route
                path="/properties"
                element={
                  <PageTransition>
                    <Properties />
                  </PageTransition>
                }
              />
              <Route
                path="/bidding-properties"
                element={
                  <PageTransition>
                    <BiddingProperties />
                  </PageTransition>
                }
              />
              <Route
                path="/auction-demo"
                element={
                  <PageTransition>
                    <AuctionDemo />
                  </PageTransition>
                }
              />
              <Route
                path="/auction-registration/:id"
                element={
                  <PageTransition>
                    <AuctionRegistration />
                  </PageTransition>
                }
              />
              <Route
                path="/property/:id"
                element={
                  <PageTransition>
                    <PropertyDetail />
                  </PageTransition>
                }
              />
              <Route
                path="/property/:id/book-visit"
                element={
                  <PageTransition>
                    <PropertyVisitBooking />
                  </PageTransition>
                }
              />
              <Route
                path="/property/:id/contact-agent"
                element={
                  <PageTransition>
                    <ContactAgentForm />
                  </PageTransition>
                }
              />
              <Route
                path="/bidding-property/:id"
                element={
                  <PageTransition>
                    <BiddingPropertyDetail />
                  </PageTransition>
                }
              />
              <Route
                path="/about"
                element={
                  <PageTransition>
                    <AboutUs />
                  </PageTransition>
                }
              />
              <Route
                path="/contact"
                element={
                  <PageTransition>
                    <ContactUs />
                  </PageTransition>
                }
              />
              <Route
                path="/docverification"
                element={
                  <PageTransition>
                    <DocumentVerification />
                  </PageTransition>
                }
              />
              <Route
                path="/docverification/results"
                element={
                  <PageTransition>
                    <DocumentVerificationResults />
                  </PageTransition>
                }
              />
              <Route
                path="/ribil-scorecard-page"
                element={
                  <PageTransition>
                    <RibilScorecardPage />
                  </PageTransition>
                }
              />
              <Route
                path="/docverification/final"
                element={
                  <PageTransition>
                    <DocumentVerificationFinal />
                  </PageTransition>
                }
              />

              {/* Service Routes - Premium Pages */}
              <Route
                path="/services/digital-vault"
                element={
                  <PageTransition>
                    <DigitalVault />
                  </PageTransition>
                }
              />
              <Route
                path="/services/document-verification"
                element={
                  <PageTransition>
                    <DocumentVerificationService />
                  </PageTransition>
                }
              />
              <Route
                path="/services/esign-estamp"
                element={
                  <PageTransition>
                    <ESignEStampServices />
                  </PageTransition>
                }
              />
              <Route
                path="/services/estamp"
                element={
                  <PageTransition>
                    <EStampPage />
                  </PageTransition>
                }
              />
              <Route
                path="/services/home-insurance"
                element={
                  <PageTransition>
                    <HomeInsuranceAssistance />
                  </PageTransition>
                }
              />
              <Route
                path="/services/home-insurance/home-buyers"
                element={
                  <PageTransition>
                    <HomeBuyersGuidance />
                  </PageTransition>
                }
              />
              <Route
                path="/services/home-insurance/property-owners"
                element={
                  <PageTransition>
                    <PropertyOwnersGuidance />
                  </PageTransition>
                }
              />
              <Route
                path="/services/home-insurance/first-time-buyers"
                element={
                  <PageTransition>
                    <FirstTimeBuyersGuidance />
                  </PageTransition>
                }
              />
              <Route
                path="/services/legal-loan-assistance"
                element={
                  <PageTransition>
                    <LegalLoanAssistancePremium />
                  </PageTransition>
                }
              />
              <Route
                path="/services/market-analysis"
                element={
                  <PageTransition>
                    <MarketAnalysisPremium />
                  </PageTransition>
                }
              />
              <Route
                path="/services/digital-marketing"
                element={
                  <PageTransition>
                    <DigitalMarketingServices />
                  </PageTransition>
                }
              />
              <Route
                path="/services/property-management"
                element={
                  <PageTransition>
                    <PropertyManagementPremium />
                  </PageTransition>
                }
              />
              <Route
                path="/services/property-valuation"
                element={
                  <PageTransition>
                    <PropertyValuation />
                  </PageTransition>
                }
              />
              <Route
                path="/services/rera-compliance"
                element={
                  <PageTransition>
                    <RERACompliancePremium />
                  </PageTransition>
                }
              />
              <Route
                path="/services/rera-guidance-buyers"
                element={
                  <PageTransition>
                    <RERAGuidanceBuyers />
                  </PageTransition>
                }
              />
              <Route
                path="/services/rera-registration-guidance"
                element={
                  <PageTransition>
                    <RERARegistrationGuidance />
                  </PageTransition>
                }
              />
              <Route
                path="/services/title-search"
                element={
                  <PageTransition>
                    <TitleSearchPremium />
                  </PageTransition>
                }
              />
              <Route
                path="/title-search/results/:searchId"
                element={
                  <PageTransition>
                    <TitleSearchResults />
                  </PageTransition>
                }
              />
              <Route
                path="/title-search/instant-results"
                element={
                  <PageTransition>
                    <TitleSearchInstantResults />
                  </PageTransition>
                }
              />

              {/* Premium Service Routes */}
              <Route
                path="/services/esign-estamp-premium"
                element={
                  <PageTransition>
                    <ESignEStampServices />
                  </PageTransition>
                }
              />
              <Route
                path="/services/property-valuation-premium"
                element={
                  <PageTransition>
                    <PropertyValuationPremium />
                  </PageTransition>
                }
              />
              <Route
                path="/services/market-analysis-premium"
                element={
                  <PageTransition>
                    <MarketAnalysisPremium />
                  </PageTransition>
                }
              />
              <Route
                path="/services/title-search-premium"
                element={
                  <PageTransition>
                    <TitleSearchPremium />
                  </PageTransition>
                }
              />
              <Route
                path="/services/rera-compliance-premium"
                element={
                  <PageTransition>
                    <RERACompliancePremium />
                  </PageTransition>
                }
              />
              <Route
                path="/services/property-management-premium"
                element={
                  <PageTransition>
                    <PropertyManagementPremium />
                  </PageTransition>
                }
              />
              <Route
                path="/services/home-insurance-premium"
                element={
                  <PageTransition>
                    <HomeInsuranceAssistance />
                  </PageTransition>
                }
              />

              {/* Contract Management Service */}
              <Route
                path="/services/contract-management"
                element={
                  <PageTransition>
                    <ContractManagement />
                  </PageTransition>
                }
              />

              {/* Fallback for any other service routes */}
              <Route
                path="/services/:serviceId"
                element={
                  <PageTransition>
                    <ServicePage />
                  </PageTransition>
                }
              />

              {/* Advisory Routes */}
              <Route
                path="/advisory/market-trends"
                element={
                  <PageTransition>
                    <MarketTrendsPage />
                  </PageTransition>
                }
              />
              <Route
                path="/advisory/rental-yield"
                element={
                  <PageTransition>
                    <RentalYieldPage />
                  </PageTransition>
                }
              />
              <Route
                path="/advisory/risk-compliance"
                element={
                  <PageTransition>
                    <RiskCompliancePage />
                  </PageTransition>
                }
              />
              <Route
                path="/advisory/nri-assistance"
                element={
                  <PageTransition>
                    <NRIAssistance />
                  </PageTransition>
                }
              />
              <Route
                path="/advisory/property-dossier"
                element={
                  <PageTransition>
                    <CustomPropertyDossier />
                  </PageTransition>
                }
              />
              <Route
                path="/advisory/:advisoryId"
                element={
                  <PageTransition>
                    <AdvisoryPage />
                  </PageTransition>
                }
              />
              <Route
                path="/advisory/investment-insights"
                element={
                  <PageTransition>
                    <InvestmentInsights />
                  </PageTransition>
                }
              />
              <Route
                path="/advisory/legal-health"
                element={
                  <PageTransition>
                    <LegalHealthLanding />
                  </PageTransition>
                }
              />
              <Route
                path="/buyer/legal-health/create-report"
                element={
                  <PageTransition>
                    <UserRoute>
                      <CreateLegalHealthReport />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/buyer/legal-health/my-reports"
                element={
                  <PageTransition>
                    <UserRoute>
                      <MyLegalHealthReports />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/buyer/legal-health/report/:id"
                element={
                  <PageTransition>
                    <UserRoute>
                      <ReportDetails />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/buyer/legal-health/upload-documents/:id"
                element={
                  <PageTransition>
                    <UserRoute>
                      <UploadDocuments />
                    </UserRoute>
                  </PageTransition>
                }
              />

              {/* Transaction Routes */}
              <Route
                path="/transactions"
                element={
                  <PageTransition>
                    <TransactionHistoryPage />
                  </PageTransition>
                }
              />
              <Route
                path="/transactions/:transactionId"
                element={
                  <PageTransition>
                    <TransactionDetails />
                  </PageTransition>
                }
              />

              {/* Privacy & Data Retention */}
              <Route
                path="/privacy-settings"
                element={
                  <PageTransition>
                    <UserRoute>
                      <PrivacySettings />
                    </UserRoute>
                  </PageTransition>
                }
              />

              {/* Solutions Routes */}
              <Route
                path="/solutions/:solutionId"
                element={
                  <PageTransition>
                    <SolutionsPage />
                  </PageTransition>
                }
              />

              {/* Digital Vault Demo */}
              <Route
                path="/digital-vault-demo"
                element={
                  <PageTransition>
                    <DigitalVaultDemo />
                  </PageTransition>
                }
              />
              <Route
                path="/list-property"
                element={
                  <PageTransition>
                    <ListingProvider>
                      <ListingForm />
                    </ListingProvider>
                  </PageTransition>
                }
              />
              <Route
                path="/map-selector"
                element={
                  <PageTransition>
                    <MapSelector />
                  </PageTransition>
                }
              />
              <Route
                path="/single-page-listing"
                element={
                  <PageTransition>
                    <SinglePageListingForm />
                  </PageTransition>
                }
              />
              <Route
                path="/emi-calculator"
                element={
                  <PageTransition>
                    <EMICalculator />
                  </PageTransition>
                }
              />
              <Route
                path="/loan-eligibility"
                element={
                  <PageTransition>
                    <LoanEligibility />
                  </PageTransition>
                }
              />
              <Route
                path="/budget-calculator"
                element={
                  <PageTransition>
                    <BudgetCalculator />
                  </PageTransition>
                }
              />
              <Route
                path="/area-converter"
                element={
                  <PageTransition>
                    <AreaConverter />
                  </PageTransition>
                }
              />
              <Route
                path="/stamp-duty-calculator"
                element={
                  <PageTransition>
                    <StampDutyCalculator />
                  </PageTransition>
                }
              />
              <Route
                path="/price-prediction"
                element={
                  <PageTransition>
                    <PricePrediction />
                  </PageTransition>
                }
              />
              <Route
                path="/valuation-reports"
                element={
                  <PageTransition>
                    <ValuationReportsPage />
                  </PageTransition>
                }
              />
              <Route
                path="/solutions/valuation-reports"
                element={
                  <PageTransition>
                    <ValuationReportsPage />
                  </PageTransition>
                }
              />
              <Route
                path="/premium"
                element={
                  <PageTransition>
                    <PremiumPackages />
                  </PageTransition>
                }
              />
              <Route
                path="/login"
                element={
                  <PageTransition>
                    <BuyerAuth />
                  </PageTransition>
                }
              />
              <Route
                path="/signup"
                element={
                  <PageTransition>
                    <BuyerAuth />
                  </PageTransition>
                }
              />
              <Route
                path="/auth/:provider/callback"
                element={
                  <PageTransition>
                    <OAuthCallback />
                  </PageTransition>
                }
              />
              <Route
                path="/auth/google/callback"
                element={
                  <PageTransition>
                    <GoogleCallback />
                  </PageTransition>
                }
              />
              <Route
                path="/forgot-password"
                element={
                  <PageTransition>
                    <ForgotPasswordPage />
                  </PageTransition>
                }
              />
              <Route
                path="/seller/forgot-password"
                element={
                  <PageTransition>
                    <SellerForgotPasswordPage />
                  </PageTransition>
                }
              />
              <Route
                path="/agent/forgot-password"
                element={
                  <PageTransition>
                    <AgentForgotPasswordPage />
                  </PageTransition>
                }
              />

              {/* KYC Routes */}
              <Route
                path="/kyc"
                element={
                  <PageTransition>
                    <KYCPage />
                  </PageTransition>
                }
              />
              <Route
                path="/owner-kyc"
                element={
                  <PageTransition>
                    <OwnerKYCForm />
                  </PageTransition>
                }
              />
              <Route
                path="/agent-kyc"
                element={
                  <PageTransition>
                    <AgentKYCForm />
                  </PageTransition>
                }
              />
              <Route
                path="/buyer-kyc"
                element={
                  <PageTransition>
                    <BuyerKYCForm />
                  </PageTransition>
                }
              />

              <Route
                path="/profile"
                element={
                  <PageTransition>
                    <Profile />
                  </PageTransition>
                }
              />
              <Route
                path="/activity"
                element={
                  <PageTransition>
                    <YourActivity />
                  </PageTransition>
                }
              />
              <Route
                path="/profile/complete"
                element={
                  <PageTransition>
                    <Suspense fallback={<ProfileSkeleton />}>
                      <ProfileComplete />
                    </Suspense>
                  </PageTransition>
                }
              />
              <Route
                path="/blog/:id"
                element={
                  <PageTransition>
                    <BlogDetail />
                  </PageTransition>
                }
              />
              <Route
                path="/legal-guidelines"
                element={
                  <PageTransition>
                    <LegalGuidelines />
                  </PageTransition>
                }
              />
              <Route
                path="/privacy-policy"
                element={
                  <PageTransition>
                    <PrivacyPolicy />
                  </PageTransition>
                }
              />
              <Route
                path="/help-center"
                element={
                  <PageTransition>
                    <HelpCenter />
                  </PageTransition>
                }
              />
              <Route
                path="/nal-stories"
                element={
                  <PageTransition>
                    <NALStories />
                  </PageTransition>
                }
              />
              <Route
                path="/ar-viewer"
                element={
                  <PageTransition>
                    <ARViewer />
                  </PageTransition>
                }
              />
              <Route
                path="/campaign-management"
                element={
                  <PageTransition>
                    <CampaignManagementPage />
                  </PageTransition>
                }
              />
              <Route
                path="/campaign-form"
                element={
                  <PageTransition>
                    <CampaignForm />
                  </PageTransition>
                }
              />
              <Route
                path="/heatmap"
                element={
                  <PageTransition>
                    <HeatmapPage />
                  </PageTransition>
                }
              />
              <Route
                path="/smart-land-maps"
                element={
                  <PageTransition>
                    <SmartLandMapsPage />
                  </PageTransition>
                }
              />
              <Route
                path="/top-markets"
                element={
                  <PageTransition>
                    <MicroMarketPerformanceHub />
                  </PageTransition>
                }
              />
              <Route path="/compare" element={<PageTransition><ComparePage /></PageTransition>} />
              <Route path="/listing-score-demo" element={<PageTransition><ListingScoreDemo /></PageTransition>} />
              <Route path="/aerial-view" element={<PageTransition><AerialView /></PageTransition>} />

              {/* Event Bus Test Route */}
              <Route
                path="/event-bus-test"
                element={
                  <PageTransition>
                    <UserRoute>
                      <EventBusTest userId={safeLocalStorage.getItem("user_id") || "test-user"} />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/negotiations/:negotiationId"
                element={
                  <PageTransition>
                    <UserRoute>
                      <NegotiationPage userRole="BUYER" />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/pricing"
                element={
                  <PageTransition>
                    <OwnerListingPackages userId={1} />
                  </PageTransition>
                }
              />
              {/* Support Ticketing Routes - Handled within dashboards */}
              {/* Grievance Management Routes */}
              <Route
                path="/grievances"
                element={
                  <PageTransition>
                    <UserRoute>
                      <GrievancePage />
                    </UserRoute>
                  </PageTransition>
                }
              />
              {/* Promotions Routes */}
              <Route
                path="/admin/coupons"
                element={
                  <PageTransition>
                    <AdminRoute>
                      <AdminCoupons />
                    </AdminRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/admin/discount-rules"
                element={
                  <PageTransition>
                    <AdminRoute>
                      <AdminDiscountRules />
                    </AdminRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/admin/scholarships"
                element={
                  <PageTransition>
                    <AdminRoute>
                      <AdminScholarships />
                    </AdminRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/scholarships"
                element={
                  <PageTransition>
                    <UserRoute>
                      <UserScholarships userId={1} userType="owner" />
                    </UserRoute>
                  </PageTransition>
                }
              />
              {/* Entitlement Engine Routes */}
              <Route
                path="/my-entitlements"
                element={
                  <PageTransition>
                    <UserRoute>
                      <UserEntitlementsDashboard userId={1} userType="owner" />
                    </UserRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/admin/entitlements"
                element={
                  <PageTransition>
                    <AdminRoute>
                      <AdminEntitlementManager />
                    </AdminRoute>
                  </PageTransition>
                }
              />
              <Route
                path="/"
                element={
                  <PageTransition>
                    <HomePage />
                  </PageTransition>
                }
              />
            </Routes>
          </AnimatePresence>
        </main>
        {!isDashboardRoute && !isAdminRoute && !isAuthRoute && (
          <SafeWrapper componentName="Footer">
            <Footer />
          </SafeWrapper>
        )}
        <SafeWrapper componentName="Chatbot">
          <Chatbot />
        </SafeWrapper>
        <SafeWrapper componentName="WhatsAppFloat">
          <WhatsAppFloat />
        </SafeWrapper>
        <SafeWrapper componentName="CompareBar">
          <CompareBar />
        </SafeWrapper>
        <Toaster
          position="top-right"
          richColors
          expand={true}
          visibleToasts={5}
          gap={8}
          offset={16}
          toastOptions={{
            style: {
              background: "white",
              border: "1px solid #e5e7eb",
              borderRadius: "12px",
              boxShadow:
                "0 10px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04)",
              padding: "16px",
              fontSize: "14px",
              fontWeight: "500",
              zIndex: 999999,
            },
            classNames: {
              toast:
                "group toast group-[.toaster]:bg-white group-[.toaster]:text-gray-950 group-[.toaster]:border-gray-200 group-[.toaster]:shadow-lg transform transition-all duration-300 ease-out",
              description: "group-[.toast]:text-gray-600 text-sm font-normal",
              actionButton:
                "group-[.toast]:bg-gray-900 group-[.toast]:text-gray-50",
              success:
                "group-[.toaster]:border-green-200 group-[.toaster]:bg-green-50",
              error:
                "group-[.toaster]:border-red-200 group-[.toaster]:bg-red-50",
              warning:
                "group-[.toaster]:border-yellow-200 group-[.toaster]:bg-yellow-50",
              info: "group-[.toaster]:border-blue-200 group-[.toaster]:bg-blue-50",
            },
            duration: 3000,
          }}
        />
      </div>
    </>
  );
}

export default function App() {
  useEffect(() => {
    safeCall(() => {
      const accessToken = safeLocalStorage.getItem("access_token");
      const agentToken = safeLocalStorage.getItem("agentToken");
      const sellerToken = safeLocalStorage.getItem("sellerToken");

      // Only auto-redirect admin if they haven't intentionally visited homepage
      if (
        accessToken &&
        window.location.pathname === "/" &&
        !sessionStorage.getItem("admin_visited_homepage")
      ) {
        try {
          const payload = JSON.parse(atob(accessToken.split(".")[1]));
          if (payload.is_admin === true) {
            window.location.href = "/admin/dashboard";
            return;
          }
        } catch (e) {
          console.warn("Could not parse token for admin check:", e);
        }
      }

      if (accessToken || agentToken || sellerToken) {
        startTokenRefreshChecker();
        initializeActivityTracking();
      }
    });

    return () => {
      cleanupActivityTracking();
    };
  }, []);

  // Get user info for Event Bus
  const getUserInfo = () => {
    try {
      const accessToken = safeLocalStorage.getItem("access_token");
      if (accessToken) {
        const payload = JSON.parse(atob(accessToken.split(".")[1]));
        return {
          userId: payload.user_id || payload.sub || "anonymous",
          token: accessToken
        };
      }
      return { userId: "anonymous", token: undefined };
    } catch {
      return { userId: "anonymous", token: undefined };
    }
  };

  const { userId, token } = getUserInfo();

  return (
    <ErrorBoundary>
      <LoadingProvider>
        <CompareProvider>
          <EventBusProvider userId={userId} token={token}>
            <Router future={{ v7_startTransition: true, v7_relativeSplatPath: true }}>
              <ScrollToTop />
              <AppContent />
            </Router>
          </EventBusProvider>
        </CompareProvider>
      </LoadingProvider>
    </ErrorBoundary>
  );
}
