const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8090';

export interface DuplicateProperty {
  property_id: number;
  title: string;
  address: string;
  city: string;
  sale_price: number;
  similarity_score: number;
}

export interface DuplicateCheckResponse {
  has_duplicates: boolean;
  duplicates: DuplicateProperty[];
  total_found: number;
  confidence_score: number;
}

export interface DuplicateReview {
  id: number;
  similarity_score: number;
  created_at: string;
  original: {
    id: number;
    title: string;
    city: string;
    price: number;
    listing_type: string;
    created_at: string;
  };
  duplicate: {
    id: number;
    title: string;
    city: string;
    price: number;
    listing_type: string;
    created_at: string;
  };
}

export interface DuplicateReviewsResponse {
  reviews: DuplicateReview[];
  total_count: number;
  page: number;
  limit: number;
  total_pages: number;
}

export const duplicateDetectionAPI = {
  checkDuplicate: async (propertyData: any): Promise<DuplicateCheckResponse> => {
    console.log('🌐 Making API request to:', `${API_BASE_URL}/api/duplicate-detection/check`);
    console.log('📤 Request payload:', propertyData);
    
    try {
      const response = await fetch(`${API_BASE_URL}/api/duplicate-detection/check`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(propertyData)
      });
      
      console.log('📥 Response status:', response.status);
      
      if (!response.ok) {
        const errorText = await response.text();
        console.error('❌ API Error:', errorText);
        throw new Error(`Failed to check duplicates: ${response.status}`);
      }
      
      const result = await response.json();
      console.log('✅ API Success:', result);
      return result;
    } catch (error) {
      console.error('❌ API Request failed:', error);
      throw error;
    }
  },

  getPendingReviews: async (page = 1, limit = 20): Promise<DuplicateReviewsResponse> => {
    console.log('📄 Fetching pending reviews...');
    const response = await fetch(`${API_BASE_URL}/api/duplicate-detection/reviews?page=${page}&limit=${limit}`);
    if (!response.ok) throw new Error('Failed to get pending reviews');
    const result = await response.json();
    console.log('📄 Pending reviews:', result);
    return result;
  },

  approveReview: async (reviewId: number, adminEmail: string) => {
    const response = await fetch(`${API_BASE_URL}/api/duplicate-detection/reviews/${reviewId}/approve`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ admin_email: adminEmail })
    });
    if (!response.ok) throw new Error('Failed to approve review');
    return response.json();
  },

  rejectReview: async (reviewId: number, adminEmail: string) => {
    const response = await fetch(`${API_BASE_URL}/api/duplicate-detection/reviews/${reviewId}/reject`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ admin_email: adminEmail })
    });
    if (!response.ok) throw new Error('Failed to reject review');
    return response.json();
  },

  mergeProperties: async (reviewId: number, adminEmail: string) => {
    const response = await fetch(`${API_BASE_URL}/api/duplicate-detection/reviews/${reviewId}/merge`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ admin_email: adminEmail })
    });
    if (!response.ok) throw new Error('Failed to merge properties');
    return response.json();
  }
};