import { refreshToken } from './auth';

const API_BASE_URL = 'http://localhost:8090';

// Admin API Interfaces
export interface AdminStats {
  total_users: number;
  total_properties: number;
  active_listings: number;
  pending_approvals: number;
  monthly_signups: number;
  revenue: number;
}

export interface UserStats {
  total_users: number;
  active_users: number;
  new_users_today: number;
  verified_users: number;
}

export interface PropertyStats {
  total_properties: number;
  active_listings: number;
  sold_properties: number;
  pending_approvals: number;
}

export interface AdminUser {
  id: string;
  username: string;
  email: string;
  phone_number: string;
  role: string;
  status: string;
  created_at: string;
  last_login: string;
}

export interface AdminProperty {
  id: string;
  title: string;
  type: string;
  status: string;
  price: number;
  location: string;
  owner: string;
  created_at: string;
}

// Utility function for admin requests with proper authentication
const authenticatedAdminFetch = async (url: string, options: RequestInit = {}): Promise<Response> => {
  const accessToken = localStorage.getItem('access_token');
  
  if (!accessToken) {
    throw new Error('No access token available');
  }

  const headers = {
    ...options.headers,
    'Content-Type': 'application/json',
    'Authorization': `Bearer ${accessToken}`
  };

  let response = await fetch(url, { ...options, headers });
  
  // If 401, try to refresh token and retry
  if (response.status === 401) {
    const refreshed = await refreshToken();
    if (refreshed) {
      const newAccessToken = localStorage.getItem('access_token');
      const newHeaders = {
        ...options.headers,
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${newAccessToken}`
      };
      response = await fetch(url, { ...options, headers: newHeaders });
    } else {
      // Refresh failed, redirect to login
      window.location.href = '/login';
      throw new Error('Authentication failed');
    }
  }
  
  return response;
};

// Admin Dashboard APIs (Placeholder implementations)

export const getAdminUsers = async (): Promise<AdminUser[]> => {
  try {
    const response = await authenticatedAdminFetch(`${API_BASE_URL}/api/admin/users`);
    
    if (!response.ok) {
      throw new Error('Failed to fetch users');
    }
    
    return await response.json();
  } catch (error) {
    // Placeholder data for development
    console.warn('Admin users API not implemented, returning mock data');
    return [];
  }
};

export const getAdminProperties = async (): Promise<AdminProperty[]> => {
  try {
    const response = await authenticatedAdminFetch(`${API_BASE_URL}/api/admin/properties`);
    
    if (!response.ok) {
      throw new Error('Failed to fetch properties');
    }
    
    return await response.json();
  } catch (error) {
    // Placeholder data for development
    console.warn('Admin properties API not implemented, returning mock data');
    return [];
  }
};

export const getUserStats = async (): Promise<UserStats> => {
  try {
    const response = await authenticatedAdminFetch(`${API_BASE_URL}/api/admin/users/stats`);
    
    if (!response.ok) {
      throw new Error('Failed to fetch user stats');
    }
    
    return await response.json();
  } catch (error) {
    // Placeholder data for development
    console.warn('User stats API not implemented, returning mock data');
    return {
      total_users: 1250,
      active_users: 890,
      new_users_today: 12,
      verified_users: 1100
    };
  }
};

export const getPropertyStats = async (): Promise<PropertyStats> => {
  try {
    const response = await authenticatedAdminFetch(`${API_BASE_URL}/api/admin/properties/stats`);
    
    if (!response.ok) {
      throw new Error('Failed to fetch property stats');
    }
    
    return await response.json();
  } catch (error) {
    // Placeholder data for development
    console.warn('Property stats API not implemented, returning mock data');
    return {
      total_properties: 850,
      active_listings: 420,
      sold_properties: 380,
      pending_approvals: 15
    };
  }
};

// User Management API
export interface AdminUserResponse {
  user_id: string;
  phone_number: string;
  first_name: string;
  last_name: string;
  email: string;
  is_verified: boolean;
  profile_completion_percentage: number;
  status: string;
  created_at: string;
  last_login: string;
}

export interface AdminPaginatedUsersResponse {
  users: AdminUserResponse[];
  total_count: number;
  page: number;
  limit: number;
  total_pages: number;
}

export interface GetUsersParams {
  page?: number;
  limit?: number;
  search?: string;
  status_filter?: string;
  verified_filter?: boolean;
}

export const getAdminUsersList = async (params: GetUsersParams = {}): Promise<AdminPaginatedUsersResponse> => {
  const queryParams = new URLSearchParams();
  
  if (params.page) queryParams.append('page', params.page.toString());
  if (params.limit) queryParams.append('limit', params.limit.toString());
  if (params.search) queryParams.append('search', params.search);
  if (params.status_filter) queryParams.append('status_filter', params.status_filter);
  if (params.verified_filter !== undefined) queryParams.append('verified_filter', params.verified_filter.toString());
  
  const url = `${API_BASE_URL}/api/admin/users${queryParams.toString() ? '?' + queryParams.toString() : ''}`;
  
  try {
    const response = await authenticatedAdminFetch(url);
    
    if (!response.ok) {
      throw new Error('Failed to fetch users list');
    }
    
    return await response.json();
  } catch (error) {
    console.error('Error fetching users list:', error);
    throw error;
  }
};

// Property Management API
export interface AdminPropertyResponse {
  id: number;
  title: string;
  listing_type: string;
  property_type: string;
  status: string;
  city: string;
  sale_price: number;
  monthly_rent: number;
  listed_by_id: string;
  views: number;
  inquiries: number;
  created_at: string;
}

export interface AdminPaginatedPropertiesResponse {
  properties: AdminPropertyResponse[];
  total_count: number;
  page: number;
  limit: number;
  total_pages: number;
}

export interface GetPropertiesParams {
  page?: number;
  limit?: number;
  search?: string;
  listing_type_filter?: string;
  property_type_filter?: string;
  status_filter?: string;
}

export const getAdminPropertiesList = async (params: GetPropertiesParams = {}): Promise<AdminPaginatedPropertiesResponse> => {
  const queryParams = new URLSearchParams();
  
  if (params.page) queryParams.append('page', params.page.toString());
  if (params.limit) queryParams.append('limit', params.limit.toString());
  if (params.search) queryParams.append('search', params.search);
  if (params.listing_type_filter) queryParams.append('listing_type_filter', params.listing_type_filter);
  if (params.property_type_filter) queryParams.append('property_type_filter', params.property_type_filter);
  if (params.status_filter) queryParams.append('status_filter', params.status_filter);
  
  const url = `${API_BASE_URL}/api/admin/properties${queryParams.toString() ? '?' + queryParams.toString() : ''}`;
  
  try {
    const response = await authenticatedAdminFetch(url);
    
    if (!response.ok) {
      throw new Error('Failed to fetch properties list');
    }
    
    return await response.json();
  } catch (error) {
    console.error('Error fetching properties list:', error);
    throw error;
  }
};

// Dashboard Statistics API
export interface AdminUserStats {
  total_users: number;
  active_users: number;
  verified_users: number;
  users_with_complete_profile: number;
  new_users_today: number;
  new_users_this_week: number;
  new_users_this_month: number;
  users_by_role: Record<string, number>;
}

export interface AdminPropertyStats {
  total_properties: number;
  active_properties: number;
  sold_properties: number;
  rented_properties: number;
  properties_by_type: Record<string, number>;
  properties_by_city: Record<string, number>;
  average_price: number;
  new_properties_today: number;
  new_properties_this_week: number;
  new_properties_this_month: number;
}

export interface AdminDashboardStats {
  user_stats: AdminUserStats;
  property_stats: AdminPropertyStats;
  generated_at: string;
}

export const getAdminDashboardStats = async (): Promise<AdminDashboardStats> => {
  try {
    const response = await authenticatedAdminFetch(`${API_BASE_URL}/api/admin/dashboard/stats`);
    
    if (!response.ok) {
      console.error('API Response:', response.status, response.statusText);
      throw new Error(`Failed to fetch dashboard stats: ${response.status}`);
    }
    
    const data = await response.json();
    console.log('Dashboard Stats API Response:', data);
    return data;
  } catch (error) {
    console.error('Error fetching dashboard stats:', error);
    const [userStats, propertyStats] = await Promise.all([
      getAdminUserStats(),
      getAdminPropertyStats()
    ]);
    return {
      user_stats: userStats,
      property_stats: propertyStats,
      generated_at: new Date().toISOString()
    };
  }
};

// Role Management API
export const updateUserRole = async (userId: string, role: 'admin' | 'user'): Promise<string> => {
  try {
    console.log('Sending role update:', { user_id: userId, role: role });
    
    const response = await fetch('/api/roles/', {
      method: 'PUT',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        user_id: userId,
        role: role
      })
    });
    
    if (!response.ok) {
      const errorText = await response.text();
      console.error('Role update error:', response.status, errorText);
      throw new Error(`Failed to update user role: ${response.status} - ${errorText}`);
    }
    
    return await response.json();
  } catch (error) {
    console.error('Error updating user role:', error);
    throw error;
  }
};

export const getAdminUserStats = async (): Promise<AdminUserStats> => {
  try {
    const response = await authenticatedAdminFetch(`${API_BASE_URL}/api/admin/users/stats`);
    
    if (!response.ok) {
      console.error('API Response:', response.status, response.statusText);
      throw new Error(`Failed to fetch user stats: ${response.status}`);
    }
    
    const data = await response.json();
    console.log('User Stats API Response:', data);
    return data;
  } catch (error) {
    console.error('Error fetching user stats:', error);
    return {
      total_users: 0,
      active_users: 0,
      verified_users: 0,
      users_with_complete_profile: 0,
      new_users_today: 0,
      new_users_this_week: 0,
      new_users_this_month: 0,
      users_by_role: {}
    };
  }
};

export const getAdminPropertyStats = async (): Promise<AdminPropertyStats> => {
  try {
    const response = await authenticatedAdminFetch(`${API_BASE_URL}/api/admin/properties/stats`);
    
    if (!response.ok) {
      console.error('API Response:', response.status, response.statusText);
      throw new Error(`Failed to fetch property stats: ${response.status}`);
    }
    
    const data = await response.json();
    console.log('Property Stats API Response:', data);
    return data;
  } catch (error) {
    console.error('Error fetching property stats:', error);
    return {
      total_properties: 0,
      active_properties: 0,
      sold_properties: 0,
      rented_properties: 0,
      properties_by_type: {},
      properties_by_city: {},
      average_price: 0,
      new_properties_today: 0,
      new_properties_this_week: 0,
      new_properties_this_month: 0
    };
  }
};

// Pending Properties API
export const getPendingProperties = async (params: GetPropertiesParams = {}): Promise<AdminPaginatedPropertiesResponse> => {
  const queryParams = new URLSearchParams();
  
  if (params.page) queryParams.append('page', params.page.toString());
  if (params.limit) queryParams.append('limit', params.limit.toString());
  if (params.search) queryParams.append('search', params.search);
  
  const url = `${API_BASE_URL}/api/admin/properties/pending${queryParams.toString() ? '?' + queryParams.toString() : ''}`;
  
  try {
    const response = await authenticatedAdminFetch(url);
    
    if (!response.ok) {
      throw new Error('Failed to fetch pending properties');
    }
    
    return await response.json();
  } catch (error) {
    console.error('Error fetching pending properties:', error);
    throw error;
  }
};

// Property Approval API
export const approveProperty = async (propertyId: number, message?: string): Promise<void> => {
  try {
    const response = await authenticatedAdminFetch(`${API_BASE_URL}/api/admin/properties/${propertyId}/approve`, {
      method: 'POST',
      body: JSON.stringify({ message })
    });
    
    if (!response.ok) {
      throw new Error('Failed to approve property');
    }
  } catch (error) {
    console.error('Error approving property:', error);
    throw error;
  }
};

// Property Rejection API
export const rejectProperty = async (propertyId: number, reason: string): Promise<void> => {
  try {
    const response = await authenticatedAdminFetch(`${API_BASE_URL}/api/admin/properties/${propertyId}/reject`, {
      method: 'POST',
      body: JSON.stringify({ reason })
    });
    
    if (!response.ok) {
      throw new Error('Failed to reject property');
    }
  } catch (error) {
    console.error('Error rejecting property:', error);
    throw error;
  }
};