// Admin Property Management API functions
const API_BASE_URL = 'http://localhost:8090';

// Generic API request function
const apiRequest = async (endpoint: string, options: RequestInit = {}) => {
  const token = localStorage.getItem('access_token');
  const response = await fetch(`${API_BASE_URL}${endpoint}`, {
    headers: {
      'Content-Type': 'application/json',
      ...(token && { Authorization: `Bearer ${token}` }),
      ...options.headers,
    },
    ...options,
  });

  if (!response.ok) {
    throw new Error(`API Error: ${response.status}`);
  }

  return response.json();
};

export interface PendingProperty {
  id: string;
  title: string;
  listing_type: string;
  property_type: string;
  city: string;
  sale_price: number;
  monthly_rent: number;
  status: 'pending' | 'approved' | 'rejected';
  created_at: string;
  views: number;
  inquiries: number;
  listed_by: {
    id: number;
    username: string;
    email: string;
    first_name: string;
    last_name: string;
  };
}

export interface PendingPropertiesResponse {
  properties: PendingProperty[];
  total_count: number;
  total_pages: number;
  current_page: number;
  page_size: number;
}

export interface ApprovePropertyRequest {
  approved_by: string;
}

export interface RejectPropertyRequest {
  rejection_reason: string;
  rejected_by: string;
}

// Get pending properties
export const getPendingProperties = async (params: {
  page?: number;
  limit?: number;
}): Promise<PendingPropertiesResponse> => {
  const queryParams = new URLSearchParams({
    page: (params.page || 1).toString(),
    limit: (params.limit || 20).toString(),
    status: 'pending' // Filter for pending status
  });

  try {
    // Try the specific admin endpoint first
    return await apiRequest(`/api/admin/properties/pending?${queryParams}`, {
      method: 'GET',
    });
  } catch (error) {
    console.log('Admin pending endpoint not available, trying general properties endpoint');
    
    // Fallback to general properties endpoint with pending status filter
    try {
      const response = await fetch(`${API_BASE_URL}/api/properties?${queryParams}`);
      if (response.ok) {
        const data = await response.json();
        // Filter for pending properties if the backend doesn't support status filtering
        const allProperties = data.properties || [];
        const pendingProperties = allProperties.filter((property: any) => 
          property.status === 'pending' || property.status === 'under_review'
        );
        
        // Transform the response to match expected format
        return {
          properties: pendingProperties,
          total_count: pendingProperties.length,
          total_pages: Math.ceil(pendingProperties.length / (params.limit || 20)),
          current_page: params.page || 1,
          page_size: params.limit || 20
        };
      }
    } catch (fallbackError) {
      console.log('General properties endpoint also failed');
    }
    
    // Return empty result if both endpoints fail
    return {
      properties: [],
      total_count: 0,
      total_pages: 0,
      current_page: 1,
      page_size: 20
    };
  }
};

// Approve property
export const approveProperty = async (
  propertyId: string,
  data: ApprovePropertyRequest
): Promise<any> => {
  console.log('=== API APPROVAL DEBUG START ===');
  console.log('Property ID:', propertyId);
  console.log('Approval data:', data);
  console.log('JSON stringified data:', JSON.stringify(data));
  console.log('API URL:', `${API_BASE_URL}/api/admin/properties/${propertyId}/approve`);
  
  const result = await apiRequest(`/api/admin/properties/${propertyId}/approve`, {
    method: 'POST',
    body: JSON.stringify(data),
  });
  
  console.log('Approval API result:', result);
  console.log('=== API APPROVAL DEBUG END ===');
  return result;
};

// Reject property
export const rejectProperty = async (
  propertyId: string,
  data: RejectPropertyRequest
): Promise<any> => {
  console.log('=== API REJECTION DEBUG START ===');
  console.log('Property ID:', propertyId);
  console.log('Rejection data:', data);
  console.log('JSON stringified data:', JSON.stringify(data));
  console.log('API URL:', `${API_BASE_URL}/api/admin/properties/${propertyId}/reject`);
  
  const result = await apiRequest(`/api/admin/properties/${propertyId}/reject`, {
    method: 'POST',
    body: JSON.stringify(data),
  });
  
  console.log('Rejection API result:', result);
  console.log('=== API REJECTION DEBUG END ===');
  return result;
};