
import axios, { InternalAxiosRequestConfig, AxiosError } from 'axios';

// Base URL for admin API
const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:8090';

// Create axios instance with auth header
const adminAxios = axios.create({
    baseURL: API_URL,
    headers: {
        'Content-Type': 'application/json',
    },
});

// Helper function to find valid token from all possible sources
const getToken = (): string | null => {
    // Check all possible token storage keys
    const tokenKeys = [
        'access_token', // Prioritize system-standard key used by auth.ts
        'admin_token',
        'token',
        'sellerToken',
        'agentToken'
    ];

    for (const key of tokenKeys) {
        const value = localStorage.getItem(key);

        // Strictly check for JWT format (header.payload.signature)
        if (value && value.length > 20 && !value.startsWith('{') && value.split('.').length === 3) {
            console.log(`[AdminService] Using valid JWT from: ${key}`);
            return value;
        }

        // Handle JSON-wrapped tokens
        if (value && value.startsWith('{')) {
            try {
                const parsed = JSON.parse(value);
                const candidates = [parsed.token, parsed.access_token];

                for (const candidate of candidates) {
                    if (candidate && typeof candidate === 'string' && candidate.split('.').length === 3) {
                        return candidate;
                    }
                }
            } catch (e) {
                // Ignore parse errors
            }
        }
    }

    // Check user_session object
    const sessionStr = localStorage.getItem('user_session');
    if (sessionStr) {
        try {
            const session = JSON.parse(sessionStr);
            if (session.token) {
                console.log('[AdminService] Using token from: user_session');
                return session.token;
            }
            if (session.access_token) {
                console.log('[AdminService] Using token from: user_session.access_token');
                return session.access_token;
            }
        } catch (e) {
            // Ignore
        }
    }

    // Check 'user' object (used by some login flows)
    const userStr = localStorage.getItem('user');
    if (userStr) {
        try {
            const user = JSON.parse(userStr);
            if (user.token) {
                console.log('[AdminService] Using token from: user.token');
                return user.token;
            }
            if (user.access_token) {
                console.log('[AdminService] Using token from: user.access_token');
                return user.access_token;
            }
        } catch (e) {
            // Ignore
        }
    }

    console.warn('[AdminService] No valid token found in localStorage!');
    console.log('[AdminService] Available localStorage keys:', Object.keys(localStorage));
    return null;
};

// Interceptor to add token to requests
adminAxios.interceptors.request.use(
    (config: InternalAxiosRequestConfig) => {
        const token = getToken();

        if (token) {
            config.headers.set('Authorization', `Bearer ${token}`);
        }
        return config;
    },
    (error: AxiosError) => {
        if (error.response && (error.response.status === 401 || error.response.status === 403)) {
            console.warn('[AdminService] Token expired or invalid.');
        }
        return Promise.reject(error);
    }
);

export const adminService = {
    // ==========================================
    // Feature 1 & 2: API Keys Management
    // ==========================================
    apiKeys: {
        getAll: () => adminAxios.get('/api/admin/developer/api-keys'),
        list: () => adminAxios.get('/api/admin/developer/api-keys'),
        getSummary: () => adminAxios.get('/api/admin/developer/summary'),
        getById: (id: number) => adminAxios.get(`/api/admin/developer/api-keys/${id}`),
        create: (data: any) => adminAxios.post('/api/admin/developer/api-keys', data),
        update: (id: number, data: any) => adminAxios.patch(`/api/admin/developer/api-keys/${id}`, data),
        delete: (id: number) => adminAxios.delete(`/api/admin/developer/api-keys/${id}`),
        revoke: (id: number) => adminAxios.delete(`/api/admin/developer/api-keys/${id}`),
        regenerate: (id: number) => adminAxios.post(`/api/admin/developer/api-keys/${id}/regenerate`),
        activate: (id: number) => adminAxios.post(`/api/admin/developer/api-keys/${id}/activate`),
        deactivate: (id: number) => adminAxios.post(`/api/admin/developer/api-keys/${id}/deactivate`),
        getStats: (id: number) => adminAxios.get(`/api/admin/developer/api-keys/${id}/stats`),
        stats: () => adminAxios.get('/api/admin/developer/usage-stats'),
        monitorUsage: () => adminAxios.get('/api/admin/developer/usage-stats'),
        sendBundle: (email: string, keyId: number) => adminAxios.post('/api/v1/devtools/delivery/send/', { email, key_id: keyId }),
        downloadBundle: () => adminAxios.get('/api/v1/devtools/delivery/download/', { responseType: 'blob' }),
        getRawKey: (id: number) => adminAxios.get(`/api/admin/developer/api-keys/${id}/raw`),
    },

    // ==========================================
    // Feature 3: Webhook Management
    // ==========================================
    webhooks: {
        getAll: () => adminAxios.get('/api/admin/webhooks/endpoints'),
        list: () => adminAxios.get('/api/admin/webhooks/endpoints'),
        getSummary: () => adminAxios.get('/api/admin/webhooks/endpoints/summary'),
        summary: () => adminAxios.get('/api/admin/webhooks/endpoints/summary'),
        getById: (id: number) => adminAxios.get(`/api/admin/webhooks/endpoints/${id}`),
        create: (data: any) => adminAxios.post('/api/admin/webhooks/endpoints', data),
        update: (id: number, data: any) => adminAxios.patch(`/api/admin/webhooks/endpoints/${id}`, data),
        delete: (id: number) => adminAxios.delete(`/api/admin/webhooks/endpoints/${id}`),
        test: (id: number) => adminAxios.post(`/api/admin/webhooks/endpoints/${id}/test`),
        getDeliveries: (id: number, params?: any) => adminAxios.get(`/api/admin/webhooks/endpoints/${id}/delivery-logs`, { params }),
        logs: (id: number) => adminAxios.get(`/api/admin/webhooks/endpoints/${id}/delivery-logs`),
        getAllDeliveries: (params?: any) => adminAxios.get('/api/admin/webhooks/deliveries', { params }),
        getDeliveryById: (id: number) => adminAxios.get(`/api/admin/webhooks/deliveries/${id}`),
        monitorStats: () => adminAxios.get('/api/admin/webhooks/monitoring/stats'),
    },

    // ==========================================
    // Feature 4: Integration Marketplace
    // ==========================================
    integrations: {
        getAvailable: () => adminAxios.get('/api/admin/marketplace/catalog'),
        getInstalled: () => adminAxios.get('/api/admin/marketplace/install'),
        getById: (id: number) => adminAxios.get(`/api/admin/marketplace/install/${id}`),
        install: (slug: string, data: any) => adminAxios.post(`/api/admin/marketplace/install/${slug}`, data),
        uninstall: (id: number) => adminAxios.delete(`/api/admin/marketplace/install/${id}`),
        test: (id: number) => adminAxios.post(`/api/admin/marketplace/test/${id}`),
        updateSecrets: (id: number, secrets: any) => adminAxios.put(`/api/admin/marketplace/install/${id}/secrets`, secrets),
        getAuditLogs: (id: number) => id === 0 ? adminAxios.get('/api/admin/vault/audit/') : adminAxios.get(`/api/admin/vault/audit/${id}`),
        getVaultHealth: () => adminAxios.get('/api/admin/vault/status'),
        rotateVaultKeys: () => adminAxios.post('/api/admin/vault/rotate-keys'),
        monitorHealth: () => adminAxios.get('/api/admin/system/marketplace-health'),
    },

    // ==========================================
    // Connectors (New Feature)
    // ==========================================
    connectors: {
        getConnectors: () => adminAxios.get('/api/admin/connectors/'),
        getSyncJobs: (connectorId?: number | null) => adminAxios.get('/api/admin/connectors/sync-jobs/', { params: { connector_id: connectorId } }),
        triggerSync: (id: number) => adminAxios.post(`/api/admin/connectors/${id}/sync/`),
        getOAuthUrl: (slug: string) => adminAxios.get('/api/admin/connectors/oauth/url', { params: { slug } }),
        handleOAuthCallback: (data: { code: string, state: string }) => adminAxios.post('/api/admin/connectors/oauth/callback', data),
        addConfig: (data: any) => adminAxios.post('/api/admin/connectors/config', data),
    },

    // ==========================================
    // Feature 10: Vault & Audit
    // ==========================================
    audit: {
        getLogs: (params?: any) => adminAxios.get('/api/view/audit-logs', { params }), // Corrected to use view endpoint if needed, or keep as is if backend supports it. actually backend has /api/v1/audit-logs usually. Let's use the new system logs for monitoring.
        logs: () => adminAxios.get('/api/admin/audit/logs/'),
    },

    // ==========================================
    // Feature 11: Unified System Monitoring
    // ==========================================
    system: {
        getDashboardStats: () => adminAxios.get('/api/admin/system/dashboard'),
        getLogs: (limit: number = 50) => adminAxios.get('/api/admin/system/logs', { params: { limit } }),
        getMarketplaceHealth: () => adminAxios.get('/api/admin/system/marketplace-health'),
        getMarketplaceLogs: (limit: number = 50) => adminAxios.get('/api/admin/system/marketplace-logs', { params: { limit } }),
        getVaultLogs: (limit: number = 50) => adminAxios.get('/api/admin/system/vault-logs', { params: { limit } }),
    },

    users: {
        list: () => adminAxios.get('/api/users/'),
    }
};

// Compatibility export
export default adminService;
