import { refreshToken } from './auth';

const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export interface AgentProfile {
  id: number;
  user_id: string;
  agent_id: string;
  full_name: string;
  email: string;
  phone_number: string;
  license_number?: string;
  agency_name?: string;
  experience_years?: number;
  specialization: string[];
  service_areas: string[];
  bio?: string;
  profile_image_url?: string;
  website_url?: string;
  social_media_links: Record<string, string>;
  office_address?: string;
  languages_spoken?: string[];
  certifications?: string[];
  education?: string;
  commission_rate?: number;
  availability_hours?: string;
  preferred_contact_method?: string;
  business_registration_number?: string;
  status: 'active' | 'inactive' | 'suspended' | 'pending_verification';
  kyc_status: 'pending' | 'verified' | 'rejected' | 'expired';
  rating: number;
  total_reviews: number;
  total_properties: number;
  total_sales: number;
  total_earnings: number;
  created_at: string;
  updated_at: string;
}

export interface AgentProfileCreate {
  full_name: string;
  email: string;
  phone_number: string;
  license_number?: string;
  agency_name?: string;
  experience_years?: number;
  specialization?: string[];
  service_areas?: string[];
  bio?: string;
  profile_image_url?: string;
  website_url?: string;
  social_media_links?: Record<string, string>;
  office_address?: string;
  languages_spoken?: string[];
  certifications?: string[];
  education?: string;
  commission_rate?: number;
  availability_hours?: string;
  preferred_contact_method?: string;
  business_registration_number?: string;
}

const makeRequest = async (url: string, options: RequestInit = {}) => {
  const accessToken = localStorage.getItem('access_token');
  if (!accessToken) throw new Error('No access token');

  const response = await fetch(url, {
    ...options,
    headers: {
      'Authorization': `Bearer ${accessToken}`,
      'Content-Type': 'application/json',
      ...options.headers,
    },
  });

  if (response.status === 401) {
    await refreshToken();
    const newToken = localStorage.getItem('access_token');
    const retryResponse = await fetch(url, {
      ...options,
      headers: {
        'Authorization': `Bearer ${newToken}`,
        'Content-Type': 'application/json',
        ...options.headers,
      },
    });
    if (!retryResponse.ok) throw new Error('Request failed');
    return retryResponse;
  }

  if (!response.ok) throw new Error('Request failed');
  return response;
};

export const getAgentProfile = async (): Promise<AgentProfile> => {
  const response = await makeRequest(`${API_BASE_URL}/agent-profile/me`);
  return await response.json();
};

export const updateAgentProfile = async (profileData: Partial<AgentProfile>): Promise<AgentProfile> => {
  const response = await makeRequest(`${API_BASE_URL}/agent-profile/me`, {
    method: 'PUT',
    body: JSON.stringify(profileData),
  });
  return await response.json();
};

export const completeAgentProfile = async (profileData: { profile_data: AgentProfileCreate }): Promise<AgentProfile> => {
  const response = await makeRequest(`${API_BASE_URL}/agent-profile/complete`, {
    method: 'POST',
    body: JSON.stringify(profileData),
  });
  return await response.json();
};

export const getAgentProfileCompletionStatus = async (): Promise<any> => {
  const response = await makeRequest(`${API_BASE_URL}/agent-profile/completion-status`);
  return await response.json();
};