const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export interface AgentVisitSchedule {
  id: number;
  user_id: string;
  property_id: number;
  status: 'pending' | 'admin_approved' | 'seller_approved' | 'seller_rejected' | 'completed';
  created_at: string;
  updated_at: string;
  full_name: string;
  email: string;
  mobile: string;
  preferred_contact: string;
  buyer_type: string;
  preferred_date: string;
  preferred_time: string;
  alternate_date?: string;
  alternate_time?: string;
  number_of_visitors: number;
  visit_mode: string;
  pickup_assistance: boolean;
  visit_reason: string;
  comments?: string;
  property_title: string;
  property_address: string;
  property_city: string;
  video_call_enabled: boolean;
  video_call_room_id?: string;
}

export interface AgentVisitSchedulesResponse {
  visits: AgentVisitSchedule[];
  total_count: number;
  page: number;
  limit: number;
}

// Get agent visits by status
export const getAgentVisits = async (
  agentId: string,
  page: number = 1,
  limit: number = 10,
  status?: string
): Promise<AgentVisitSchedulesResponse> => {
  const agentToken = localStorage.getItem('agentToken');
  const token = localStorage.getItem('token');
  const authToken = agentToken || token;
  
  const params = new URLSearchParams({
    page: page.toString(),
    limit: limit.toString(),
    ...(status && { status })
  });

  const response = await fetch(`${API_BASE}/visit-schedules/agent/${agentId}/visits?${params}`, {
    headers: {
      ...(authToken && { 'Authorization': `Bearer ${authToken}` })
    }
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Failed to fetch agent visits');
  }

  return response.json();
};

// Get agent pending visits
export const getAgentPendingVisits = async (
  agentId: string,
  page: number = 1,
  limit: number = 10
): Promise<AgentVisitSchedulesResponse> => {
  const agentToken = localStorage.getItem('agentToken');
  const token = localStorage.getItem('token');
  const authToken = agentToken || token;
  
  const params = new URLSearchParams({
    page: page.toString(),
    limit: limit.toString()
  });

  const response = await fetch(`${API_BASE}/visit-schedules/agent/${agentId}/pending?${params}`, {
    headers: {
      ...(authToken && { 'Authorization': `Bearer ${authToken}` })
    }
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Failed to fetch agent pending visits');
  }

  return response.json();
};

// Agent approve visit
export const agentApproveVisit = async (
  visitId: number,
  agentId: string
): Promise<AgentVisitSchedule> => {
  // Temporary mock for when backend is not running
  if (API_BASE.includes('localhost:8090')) {
    try {
      const healthCheck = await fetch(`${API_BASE}/health`, { signal: AbortSignal.timeout(2000) });
    } catch {
      console.warn('⚠️ Backend not running, using mock response');
      // Return mock approved visit
      return {
        id: visitId,
        user_id: 'mock-user',
        property_id: 1,
        status: 'seller_approved',
        created_at: new Date().toISOString(),
        updated_at: new Date().toISOString(),
        full_name: 'Mock User',
        email: 'mock@example.com',
        mobile: '1234567890',
        preferred_contact: 'Call',
        buyer_type: 'Individual Buyer',
        preferred_date: new Date().toISOString().split('T')[0],
        preferred_time: '10:00 AM',
        number_of_visitors: 1,
        visit_mode: 'In-person Visit',
        pickup_assistance: false,
        visit_reason: 'Property viewing',
        property_title: 'Mock Property',
        property_address: 'Mock Address',
        property_city: 'Mock City'
      } as AgentVisitSchedule;
    }
  }
  
  const agentToken = localStorage.getItem('agentToken');
  const token = localStorage.getItem('token');
  const authToken = agentToken || token;
  
  console.log('🔑 Auth token check:', { agentToken: !!agentToken, token: !!token, authToken: !!authToken });
  console.log('🎯 API call details:', { visitId, agentId, endpoint: `${API_BASE}/visit-schedules/${visitId}/agent-approve?agent_id=${agentId}` });
  
  const response = await fetch(`${API_BASE}/visit-schedules/${visitId}/agent-approve?agent_id=${agentId}`, {
    method: 'POST',
    headers: {
      ...(authToken && { 'Authorization': `Bearer ${authToken}` })
    }
  });

  console.log('📊 Response status:', response.status, response.statusText);

  if (!response.ok) {
    const error = await response.json().catch(() => ({ message: `HTTP ${response.status}: ${response.statusText}` }));
    console.error('❌ API Error:', error);
    throw new Error(error.message || 'Failed to approve visit');
  }

  return response.json();
};

// Agent reject visit
export const agentRejectVisit = async (
  visitId: number,
  agentId: string
): Promise<AgentVisitSchedule> => {
  const agentToken = localStorage.getItem('agentToken');
  const token = localStorage.getItem('token');
  const authToken = agentToken || token;
  
  const response = await fetch(`${API_BASE}/visit-schedules/${visitId}/agent-reject?agent_id=${agentId}`, {
    method: 'POST',
    headers: {
      ...(authToken && { 'Authorization': `Bearer ${authToken}` })
    }
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Failed to reject visit');
  }

  return response.json();
};

// Update visit status (for completing visits)
export const updateAgentVisitStatus = async (
  visitId: number,
  status: string
): Promise<AgentVisitSchedule> => {
  const agentToken = localStorage.getItem('agentToken');
  const token = localStorage.getItem('token');
  const authToken = agentToken || token;
  
  const response = await fetch(`${API_BASE}/visit-schedules/${visitId}`, {
    method: 'PUT',
    headers: {
      'Content-Type': 'application/json',
      ...(authToken && { 'Authorization': `Bearer ${authToken}` })
    },
    body: JSON.stringify({ status })
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Failed to update visit status');
  }

  return response.json();
};

// Seller approve visit (for admin_approved visits)
export const sellerApproveVisit = async (
  visitId: number,
  sellerId: string
): Promise<AgentVisitSchedule> => {
  const agentToken = localStorage.getItem('agentToken');
  const token = localStorage.getItem('token');
  const authToken = agentToken || token;
  
  const response = await fetch(`${API_BASE}/visit-schedules/${visitId}/seller-approve?seller_id=${sellerId}`, {
    method: 'POST',
    headers: {
      ...(authToken && { 'Authorization': `Bearer ${authToken}` })
    }
  });

  if (!response.ok) {
    const error = await response.json().catch(() => ({ message: `HTTP ${response.status}: ${response.statusText}` }));
    throw new Error(error.message || 'Failed to approve visit');
  }

  return response.json();
};