interface AbsorptionRateResponse {
  active_properties: number;
  sold_properties: number;
  total_inventory: number;
  absorption_rate: number;
}

export const getAbsorptionRate = async (
  userId: string, 
  period: string = '1y',
  role: string = 'agent'
): Promise<AbsorptionRateResponse> => {
  console.log(`🔍 Calling absorption rate API: userId=${userId}, period=${period}, role=${role}`);
  
  const token = localStorage.getItem('agentToken') || 
                localStorage.getItem('token') || 
                localStorage.getItem('access_token');
  
  console.log(`🔐 Using token: ${token ? 'exists' : 'missing'}`);
  
  const headers: HeadersInit = {
    'Content-Type': 'application/json'
  };
  
  if (token) {
    headers['Authorization'] = `Bearer ${token}`;
  }

  const url = `http://localhost:8090/api/analytics/agent/absorption-rate?user_id=${userId}&period=${period}`;
  console.log(`📡 API URL: ${url}`);

  const response = await fetch(url, { headers });
  
  console.log(`📊 Response status: ${response.status}`);

  if (!response.ok) {
    const errorText = await response.text();
    console.error(`❌ API Error: ${response.status} - ${errorText}`);
    throw new Error(`Failed to fetch absorption rate: ${response.status} - ${errorText}`);
  }

  const data = await response.json();
  console.log(`✅ API Response:`, data);
  console.log(`📊 Agent API - Sold Properties:`, data.sold_properties);
  console.log(`📊 Agent API - Active Properties:`, data.active_properties);
  console.log(`📊 Agent API - Absorption Rate:`, data.absorption_rate);
  return data;
};

export const getAdminAbsorptionRate = async (
  period: string = '1y'
): Promise<AbsorptionRateResponse> => {
  console.log(`🔍 Calling admin absorption rate API: period=${period}`);
  
  const token = localStorage.getItem('access_token') || 
                localStorage.getItem('token') || 
                localStorage.getItem('agentToken');
  
  console.log(`🔐 Using token: ${token ? 'exists' : 'missing'}`);
  
  const headers: HeadersInit = {
    'Content-Type': 'application/json'
  };
  
  if (token) {
    headers['Authorization'] = `Bearer ${token}`;
  }

  const url = `http://localhost:8090/api/analytics/admin/absorption-rate?period=${period}`;
  console.log(`📡 Admin API URL: ${url}`);
  console.log(`📡 Headers:`, headers);
  
  try {
    const response = await fetch(url, { 
      method: 'GET',
      headers,
      mode: 'cors'
    });
    
    console.log(`📊 Admin Response status: ${response.status}`);

    if (!response.ok) {
      const errorText = await response.text();
      console.error(`❌ Admin API Error: ${response.status} - ${errorText}`);
      throw new Error(`Failed to fetch admin absorption rate: ${response.status} - ${errorText}`);
    }

    const data = await response.json();
    console.log(`✅ Admin API Response:`, data);
    console.log(`📊 Admin API - Sold Properties:`, data.sold_properties);
    console.log(`📊 Admin API - Active Properties:`, data.active_properties);
    console.log(`📊 Admin API - Total Inventory:`, data.total_inventory);
    console.log(`📊 Admin API - Absorption Rate:`, data.absorption_rate);
    
    return data;
  } catch (fetchError) {
    console.error(`❌ Fetch Error:`, fetchError);
    throw fetchError;
  }
};

export const getSellerAbsorptionRate = async (
  userId: string,
  period: string = '1y'
): Promise<AbsorptionRateResponse> => {
  console.log(`🔍 Calling seller absorption rate API: userId=${userId}, period=${period}`);
  
  const token = localStorage.getItem('sellerToken') || 
                localStorage.getItem('token') || 
                localStorage.getItem('access_token');
  
  console.log(`🔐 Using token: ${token ? 'exists' : 'missing'}`);
  
  if (!userId) {
    console.error('❌ No userId provided for seller absorption rate');
    throw new Error('User ID is required');
  }
  
  const headers: HeadersInit = {
    'Content-Type': 'application/json'
  };
  
  if (token) {
    headers['Authorization'] = `Bearer ${token}`;
  }

  const url = `http://localhost:8090/api/analytics/seller/absorption-rate?user_id=${userId}&period=${period}`;
  console.log(`📡 Seller API URL: ${url}`);
  
  const response = await fetch(url, { headers });
  
  console.log(`📊 Seller Response status: ${response.status}`);

  if (!response.ok) {
    const errorText = await response.text();
    console.error(`❌ Seller API Error: ${response.status} - ${errorText}`);
    throw new Error(`Failed to fetch seller absorption rate: ${response.status} - ${errorText}`);
  }

  const data = await response.json();
  console.log(`✅ Seller API Response:`, data);
  console.log(`📊 Seller API - Sold Properties:`, data.sold_properties);
  console.log(`📊 Seller API - Active Properties:`, data.active_properties);
  console.log(`📊 Seller API - Absorption Rate:`, data.absorption_rate);
  return data;
};