const API_BASE_URL = 'http://localhost:8090/api';

export interface AuctionSession {
  id: string;
  property_id: number;
  session_status: 'scheduled' | 'active' | 'paused' | 'completed' | 'cancelled';
  starting_bid_price: number;
  reserve_price: number;
  minimum_increment: number;
  registration_fee: number;
  current_highest_bid: number;
  winning_bidder_id: string;
  total_bids: number;
  registered_bidders: number;
  start_time: string;
  end_time: string;
  actual_end_time: string;
  extension_count: number;
  auto_extension_enabled: boolean;
  extension_time_minutes: number;
  created_at: string;
  updated_at: string;
}

export interface AuctionSessionsResponse {
  sessions: AuctionSession[];
  total_count: number;
  page: number;
  limit: number;
}

export const auctionManagementAPI = {
  getAllSessions: async (page = 1, limit = 20, statusFilter?: string): Promise<AuctionSessionsResponse> => {
    const params = new URLSearchParams({
      page: page.toString(),
      limit: limit.toString(),
    });
    
    if (statusFilter) {
      params.append('status_filter', statusFilter);
    }

    const response = await fetch(`${API_BASE_URL}/auction-management/all-sessions?${params}`, {
      headers: {
        'Authorization': `Bearer ${localStorage.getItem('access_token')}`,
        'Content-Type': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch auction sessions: ${response.status}`);
    }

    return response.json();
  },

  startAuction: async (sessionId: string): Promise<string> => {
    const response = await fetch(`${API_BASE_URL}/auction-management/${sessionId}/start`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${localStorage.getItem('access_token')}`,
        'Content-Type': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to start auction: ${response.status}`);
    }

    return response.json();
  },

  pauseAuction: async (sessionId: string): Promise<string> => {
    const response = await fetch(`${API_BASE_URL}/auction-management/${sessionId}/pause`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${localStorage.getItem('access_token')}`,
        'Content-Type': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to pause auction: ${response.status}`);
    }

    return response.json();
  },

  cancelAuction: async (sessionId: string): Promise<string> => {
    const response = await fetch(`${API_BASE_URL}/auction-management/${sessionId}/cancel`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${localStorage.getItem('access_token')}`,
        'Content-Type': 'application/json',
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to cancel auction: ${response.status}`);
    }

    return response.json();
  },

  extendAuction: async (sessionId: string, extensionMinutes: number): Promise<AuctionSession> => {
    const response = await fetch(`${API_BASE_URL}/auction-management/${sessionId}/extend`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${localStorage.getItem('access_token')}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        extension_minutes: extensionMinutes,
      }),
    });

    if (!response.ok) {
      throw new Error(`Failed to extend auction: ${response.status}`);
    }

    return response.json();
  },
};