// Auction API Integration
const API_BASE_URL = 'http://localhost:8090/api';
const AUCTION_API = `${API_BASE_URL}/auctions`;
const AUCTION_MGMT_API = `${API_BASE_URL}/auction-management`;

// Types
export interface AuctionSession {
  session_id: string;
  property_id: string;
  title: string;
  description: string;
  starting_bid: number;
  current_bid: number;
  minimum_increment: number;
  registration_fee: number;
  start_time: string;
  end_time: string;
  status: 'upcoming' | 'active' | 'paused' | 'completed' | 'cancelled';
  total_bids: number;
  registered_bidders: number;
}

export interface Bid {
  bid_id: string;
  session_id: string;
  user_id: string;
  bid_amount: number;
  bid_type: 'MANUAL' | 'AUTO';
  timestamp: string;
  is_winning: boolean;
}

export interface AutoBidSettings {
  max_bid_amount: number;
  increment_amount: number;
  is_active: boolean;
}

// Public Auction APIs
export const auctionAPI = {
  // Get active auctions
  getActiveAuctions: async (page = 1, limit = 20) => {
    try {
      const response = await fetch(`${AUCTION_API}/active?page=${page}&limit=${limit}`, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to fetch active auctions: ${response.status}`);
    } catch (error) {
      console.error('Error fetching active auctions:', error);
      throw error;
    }
  },

  // Get upcoming auctions
  getUpcomingAuctions: async (page = 1, limit = 20) => {
    try {
      const response = await fetch(`${AUCTION_API}/upcoming?page=${page}&limit=${limit}`, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to fetch upcoming auctions: ${response.status}`);
    } catch (error) {
      console.error('Error fetching upcoming auctions:', error);
      throw error;
    }
  },

  // Get auction details
  getAuctionDetails: async (sessionId: string) => {
    try {
      const response = await fetch(`${AUCTION_API}/${sessionId}/details`, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to fetch auction details: ${response.status}`);
    } catch (error) {
      console.error('Error fetching auction details:', error);
      throw error;
    }
  },

  // Get current auction status
  getCurrentStatus: async (sessionId: string) => {
    try {
      const response = await fetch(`${AUCTION_API}/${sessionId}/current-status`, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to fetch auction status: ${response.status}`);
    } catch (error) {
      console.error('Error fetching auction status:', error);
      throw error;
    }
  },

  // Get public bid history
  getBidHistory: async (sessionId: string) => {
    try {
      const response = await fetch(`${AUCTION_API}/${sessionId}/bid-history`, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to fetch bid history: ${response.status}`);
    } catch (error) {
      console.error('Error fetching bid history:', error);
      throw error;
    }
  }
};

// User Auction APIs (require authentication)
export const userAuctionAPI = {
  // Register for auction
  registerForAuction: async (sessionId: string) => {
    try {
      // Get auth token from localStorage or wherever it's stored
      const token = localStorage.getItem('access_token') || sessionStorage.getItem('access_token');
      
      const headers: any = { 
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      };
      
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }
      
      const response = await fetch(`${AUCTION_API}/${sessionId}/register`, {
        method: 'POST',
        headers,
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.detail || `Failed to register for auction: ${response.status}`);
    } catch (error) {
      console.error('Error registering for auction:', error);
      throw error;
    }
  },

  // Check registration status
  getRegistrationStatus: async (sessionId: string) => {
    try {
      // Get auth token from localStorage or wherever it's stored
      const token = localStorage.getItem('access_token') || sessionStorage.getItem('access_token');
      
      const headers: any = { 'Accept': 'application/json' };
      
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }
      
      const response = await fetch(`${AUCTION_API}/${sessionId}/registration-status`, {
        method: 'GET',
        headers,
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.detail || `Failed to check registration status: ${response.status}`);
    } catch (error) {
      console.error('Error checking registration status:', error);
      throw error;
    }
  },

  // Pay registration fee
  payRegistrationFee: async (sessionId: string, paymentData: any) => {
    try {
      // Get auth token from localStorage or wherever it's stored
      const token = localStorage.getItem('access_token') || sessionStorage.getItem('access_token');
      
      const headers: any = { 
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      };
      
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }
      
      const response = await fetch(`${AUCTION_API}/${sessionId}/pay-registration-fee`, {
        method: 'POST',
        headers,
        body: JSON.stringify(paymentData),
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.detail || `Failed to pay registration fee: ${response.status}`);
    } catch (error) {
      console.error('Error paying registration fee:', error);
      throw error;
    }
  },

  // Place a bid
  placeBid: async (sessionId: string, bidAmount: number, bidType: 'MANUAL' | 'AUTO' = 'MANUAL') => {
    try {
      // Get auth token from localStorage or wherever it's stored
      const token = localStorage.getItem('access_token') || sessionStorage.getItem('access_token');
      
      const headers: any = { 
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      };
      
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }
      
      const response = await fetch(`${AUCTION_API}/${sessionId}/place-bid`, {
        method: 'POST',
        headers,
        body: JSON.stringify({ 
          bid_amount: bidAmount, 
          bid_type: bidType 
        }),
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.detail || `Failed to place bid: ${response.status}`);
    } catch (error) {
      console.error('Error placing bid:', error);
      throw error;
    }
  },

  // Get user's bids
  getMyBids: async (sessionId: string) => {
    try {
      const response = await fetch(`${AUCTION_API}/${sessionId}/my-bids`, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to fetch user bids: ${response.status}`);
    } catch (error) {
      console.error('Error fetching user bids:', error);
      throw error;
    }
  },

  // Set auto-bid settings
  setAutoBidSettings: async (sessionId: string, settings: AutoBidSettings) => {
    try {
      const response = await fetch(`${AUCTION_API}/${sessionId}/auto-bid-settings`, {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        body: JSON.stringify(settings),
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to set auto-bid settings: ${response.status}`);
    } catch (error) {
      console.error('Error setting auto-bid settings:', error);
      throw error;
    }
  }
};

// Admin Auction Management APIs
export const adminAuctionAPI = {
  // Create new auction
  createAuction: async (auctionData: any) => {
    try {
      const response = await fetch(`${AUCTION_MGMT_API}/create-session`, {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        body: JSON.stringify(auctionData),
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to create auction: ${response.status}`);
    } catch (error) {
      console.error('Error creating auction:', error);
      throw error;
    }
  },

  // Get all auctions (admin)
  getAllAuctions: async () => {
    try {
      const response = await fetch(`${AUCTION_MGMT_API}/all-sessions`, {
        method: 'GET',
        headers: { 'Accept': 'application/json' },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to fetch all auctions: ${response.status}`);
    } catch (error) {
      console.error('Error fetching all auctions:', error);
      throw error;
    }
  },

  // Start auction
  startAuction: async (sessionId: string) => {
    try {
      const response = await fetch(`${AUCTION_MGMT_API}/${sessionId}/start`, {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to start auction: ${response.status}`);
    } catch (error) {
      console.error('Error starting auction:', error);
      throw error;
    }
  },

  // Pause auction
  pauseAuction: async (sessionId: string) => {
    try {
      const response = await fetch(`${AUCTION_MGMT_API}/${sessionId}/pause`, {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to pause auction: ${response.status}`);
    } catch (error) {
      console.error('Error pausing auction:', error);
      throw error;
    }
  },

  // Cancel auction
  cancelAuction: async (sessionId: string) => {
    try {
      const response = await fetch(`${AUCTION_MGMT_API}/${sessionId}/cancel`, {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to cancel auction: ${response.status}`);
    } catch (error) {
      console.error('Error cancelling auction:', error);
      throw error;
    }
  },

  // Extend auction time
  extendAuction: async (sessionId: string, extensionData: any) => {
    try {
      const response = await fetch(`${AUCTION_MGMT_API}/${sessionId}/extend`, {
        method: 'POST',
        headers: { 
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        body: JSON.stringify(extensionData),
        mode: 'cors',
        credentials: 'omit'
      });
      
      if (response.ok) {
        return await response.json();
      }
      throw new Error(`Failed to extend auction: ${response.status}`);
    } catch (error) {
      console.error('Error extending auction:', error);
      throw error;
    }
  }
};

// Real-time auction updates (WebSocket placeholder)
export class AuctionWebSocket {
  private ws: WebSocket | null = null;
  private sessionId: string;
  private callbacks: { [event: string]: Function[] } = {};

  constructor(sessionId: string) {
    this.sessionId = sessionId;
  }

  connect() {
    try {
      // WebSocket connection for real-time updates
      this.ws = new WebSocket(`ws://localhost:8090/ws/auction/${this.sessionId}/`);
      
      this.ws.onopen = () => {
        console.log('Auction WebSocket connected');
        this.emit('connected');
      };

      this.ws.onmessage = (event) => {
        const data = JSON.parse(event.data);
        this.emit(data.type, data);
      };

      this.ws.onclose = () => {
        console.log('Auction WebSocket disconnected');
        this.emit('disconnected');
      };

      this.ws.onerror = (error) => {
        console.error('Auction WebSocket error:', error);
        this.emit('error', error);
      };
    } catch (error) {
      console.error('Failed to connect to auction WebSocket:', error);
    }
  }

  disconnect() {
    if (this.ws) {
      this.ws.close();
      this.ws = null;
    }
  }

  on(event: string, callback: Function) {
    if (!this.callbacks[event]) {
      this.callbacks[event] = [];
    }
    this.callbacks[event].push(callback);
  }

  off(event: string, callback: Function) {
    if (this.callbacks[event]) {
      this.callbacks[event] = this.callbacks[event].filter(cb => cb !== callback);
    }
  }

  private emit(event: string, data?: any) {
    if (this.callbacks[event]) {
      this.callbacks[event].forEach(callback => callback(data));
    }
  }
}

// Utility functions
export const auctionUtils = {
  // Format currency for display
  formatCurrency: (amount: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0
    }).format(amount);
  },

  // Calculate time remaining
  getTimeRemaining: (endTime: string) => {
    const now = new Date().getTime();
    const end = new Date(endTime).getTime();
    const difference = end - now;

    if (difference <= 0) {
      return { expired: true, display: 'Auction Ended' };
    }

    const days = Math.floor(difference / (1000 * 60 * 60 * 24));
    const hours = Math.floor((difference % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
    const minutes = Math.floor((difference % (1000 * 60 * 60)) / (1000 * 60));
    const seconds = Math.floor((difference % (1000 * 60)) / 1000);

    let display = '';
    if (days > 0) display += `${days}d `;
    if (hours > 0) display += `${hours}h `;
    if (minutes > 0) display += `${minutes}m `;
    if (seconds > 0) display += `${seconds}s`;

    return { expired: false, display: display.trim(), days, hours, minutes, seconds };
  },

  // Validate bid amount
  validateBid: (amount: number, currentBid: number, minimumIncrement: number) => {
    const minimumBid = currentBid + minimumIncrement;
    return {
      isValid: amount >= minimumBid,
      minimumRequired: minimumBid,
      message: amount < minimumBid ? `Minimum bid is ₹${minimumBid.toLocaleString()}` : ''
    };
  }
};