// Authentication API functions
export const startTokenRefreshChecker = () => {
  // Token refresh logic placeholder
  console.log('Token refresh checker started');
};

// Types for forgot password
export interface ForgotPasswordRequest {
  email: string;
}

export interface ForgotPasswordResponse {
  success: boolean;
  message: string;
  expires_in?: number;
}

export interface ValidateTokenRequest {
  token: string;
}

export interface ValidateTokenResponse {
  success: boolean;
  message: string;
  user_type?: string;
  email?: string;
}

export interface ResetPasswordRequest {
  token: string;
  new_password: string;
  confirm_password: string;
}

export interface ResetPasswordResponse {
  success: boolean;
  message: string;
}

export const login = async (credentials: any) => {
  // Login API call placeholder
  return { success: true };
};

export const logout = () => {
  // Logout logic placeholder
  localStorage.removeItem('token');
};

export const getToken = () => {
  return localStorage.getItem('token');
};

export const setToken = (token: string) => {
  localStorage.setItem('token', token);
};

export const refreshToken = async () => {
  const refreshToken = localStorage.getItem('refresh_token');
  if (!refreshToken) {
    throw new Error('No refresh token available');
  }
  
  try {
    const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
    const response = await fetch(`${API_BASE}/auth/refresh`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({ refresh_token: refreshToken }),
    });
    
    if (!response.ok) {
      throw new Error('Token refresh failed');
    }
    
    const data = await response.json();
    localStorage.setItem('access_token', data.access_token);
    return data;
  } catch (error) {
    localStorage.removeItem('access_token');
    localStorage.removeItem('refresh_token');
    throw error;
  }
};

export const sendOTP = async (phoneNumber: string) => {
  console.log('🔄 Sending phone OTP request:', { phone_number: phoneNumber });
  const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
  const response = await fetch(`${API_BASE}/auth/send-otp`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ phone_number: phoneNumber }),
  });
  
  console.log('📥 Phone OTP Response:', response.status, response.statusText);
  
  if (!response.ok) {
    const errorText = await response.text();
    console.error('❌ Phone OTP Error:', errorText);
    throw new Error(`Failed to send OTP: ${errorText}`);
  }
  return response.json();
};

export const loginUser = async (phoneNumber: string, otp: string) => {
  console.log('🔄 Phone login request:', { phone_number: phoneNumber, otp_length: otp.length });
  const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
  const response = await fetch(`${API_BASE}/auth/login`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ phone_number: phoneNumber, otp }),
  });
  
  console.log('📥 Phone Login Response:', response.status, response.statusText);
  
  if (!response.ok) {
    const errorText = await response.text();
    console.error('❌ Phone Login Error:', errorText);
    try {
      const errorData = JSON.parse(errorText);
      const message = errorData?.detail?.message || errorData?.message || 'Login failed';
      throw new Error(message);
    } catch (parseError: any) {
      if (parseError instanceof SyntaxError) {
        throw new Error('Login failed');
      }
      throw parseError;
    }
  }
  return response.json();
};

export const sendEmailOTP = async (email: string) => {
  console.log('🔄 Sending email OTP request:', { email });
  const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
  console.log('🔧 API_BASE:', API_BASE); // Debug log
  const response = await fetch(`${API_BASE}/auth/send-email-otp`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ email }),
  });
  
  console.log('📥 Email OTP Response:', response.status, response.statusText);
  
  if (!response.ok) {
    const errorText = await response.text();
    console.error('❌ Email OTP Error:', errorText);
    throw new Error(`Failed to send email OTP: ${errorText}`);
  }
  return response.json();
};

export const loginWithEmail = async (email: string, otp: string) => {
  console.log('🔄 Email login request:', { email, otp_code: otp });
  const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
  const response = await fetch(`${API_BASE}/auth/login-email`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ email, otp_code: otp }),
  });
  
  console.log('📥 Email Login Response:', response.status, response.statusText);
  
  if (!response.ok) {
    const errorText = await response.text();
    console.error('❌ Email Login Error:', errorText);
    try {
      const errorData = JSON.parse(errorText);
      const message = errorData?.detail?.message || errorData?.message || 'Email login failed';
      throw new Error(message);
    } catch (parseError: any) {
      // If JSON parsing fails but we have the original error, check if it's already parsed
      if (parseError instanceof SyntaxError) {
        throw new Error('Email login failed');
      }
      // Re-throw the error with the message we extracted
      throw parseError;
    }
  }
  return response.json();
};

export const sendRegistrationOTP = async (email: string) => {
  console.log('🔄 Sending registration OTP:', { email });
  const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
  const response = await fetch(`${API_BASE}/auth/send-registration-otp`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ email }),
  });
  
  console.log('📥 Registration OTP Response:', response.status, response.statusText);
  
  if (!response.ok) {
    const errorText = await response.text();
    console.error('❌ Registration OTP Error:', errorText);
    
    // Parse error response to check for specific error types
    try {
      const errorData = JSON.parse(errorText);
      if (errorData.detail?.includes('already exists') || errorData.detail?.includes('already registered') || 
          errorData.error?.includes('already exists') || errorData.message?.includes('already exists')) {
        throw new Error('This email is already registered. Please login instead.');
      }
    } catch (parseError) {
      // If JSON parsing fails, check the raw error text
      if (errorText.includes('already exists') || errorText.includes('already registered') || 
          errorText.includes('User with this email already exists')) {
        throw new Error('This email is already registered. Please login instead.');
      }
    }
    
    throw new Error(`Failed to send registration OTP: ${errorText}`);
  }
  return response.json();
};

export const verifyRegistrationOTP = async (email: string, otp: string, password: string) => {
  console.log('🔄 Verifying registration OTP:', { email, otp_length: otp.length });
  const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
  const response = await fetch(`${API_BASE}/auth/verify-registration-otp`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ email, otp_code: otp, password }),
  });
  
  console.log('📥 Registration Verify Response:', response.status, response.statusText);
  
  if (!response.ok) {
    const errorText = await response.text();
    console.error('❌ Registration Verify Error:', errorText);
    try {
      const errorData = JSON.parse(errorText);
      const message = errorData?.detail?.message || errorData?.message || 'Registration failed';
      throw new Error(message);
    } catch (parseError: any) {
      if (parseError instanceof SyntaxError) {
        throw new Error('Registration failed');
      }
      throw parseError;
    }
  }
  return response.json();
};

export const loginWithPassword = async (email: string, password: string) => {
  console.log('🔄 Password login request:', { email });
  const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
  const response = await fetch(`${API_BASE}/auth/login-password`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ email, password }),
  });
  
  console.log('📥 Password Login Response:', response.status, response.statusText);
  
  
  if (!response.ok) {
    const errorText = await response.text();
    console.error('❌ Password Login Error:', errorText);
    try {
      const errorData = JSON.parse(errorText);
      const message = errorData?.detail?.message || errorData?.message || 'Login failed';
      throw new Error(message);
    } catch (parseError: any) {
      if (parseError instanceof SyntaxError) {
        throw new Error('Login failed');
      }
      throw parseError;
    }
  }
  return response.json();
};

// Forgot Password API functions
export const sendForgotPasswordEmail = async (email: string, userType: 'buyer' | 'seller' | 'agent' = 'buyer') => {
  console.log('🔄 Sending forgot password email:', { email, userType });
  
  const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
  const endpoint = userType === 'buyer' ? `${API_BASE}/password-reset/forgot-password` : `${API_BASE}/seller-agent-auth/forgot-password`;
  
  const response = await fetch(endpoint, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ email }),
  });
  
  console.log('📥 Forgot Password Response:', response.status, response.statusText);
  
  if (!response.ok) {
    const errorText = await response.text();
    console.error('❌ Forgot Password Error:', errorText);
    throw new Error(`Failed to send reset email: ${errorText}`);
  }
  return response.json();
};

export const validateResetToken = async (token: string, userType: 'buyer' | 'seller' | 'agent' = 'buyer') => {
  console.log('🔄 Validating reset token:', { token: token.substring(0, 8) + '...', userType });
  
  const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
  const endpoint = userType === 'buyer' ? `${API_BASE}/password-reset/validate-token` : `${API_BASE}/seller-agent-auth/validate-token`;
  
  const response = await fetch(endpoint, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ token }),
  });
  
  console.log('📥 Validate Token Response:', response.status, response.statusText);
  
  if (!response.ok) {
    const errorText = await response.text();
    console.error('❌ Validate Token Error:', errorText);
    throw new Error(`Token validation failed: ${errorText}`);
  }
  return response.json();
};

export const resetPassword = async (token: string, newPassword: string, confirmPassword: string, userType: 'buyer' | 'seller' | 'agent' = 'buyer') => {
  console.log('🔄 Resetting password:', { token: token.substring(0, 8) + '...', userType });
  
  const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
  const endpoint = userType === 'buyer' ? `${API_BASE}/password-reset/reset-password` : `${API_BASE}/seller-agent-auth/reset-password`;
  
  const response = await fetch(endpoint, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ 
      token, 
      new_password: newPassword, 
      confirm_password: confirmPassword 
    }),
  });
  
  console.log('📥 Reset Password Response:', response.status, response.statusText);
  
  if (!response.ok) {
    const errorText = await response.text();
    console.error('❌ Reset Password Error:', errorText);
    throw new Error(`Password reset failed: ${errorText}`);
  }
  return response.json();
};