const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export interface UserBehaviorInsights {
  user_id: string;
  total_events: number;
  most_active_hours: number[];
  preferred_device: string;
  engagement_score: number;
  top_searched_locations: string[];
  preferred_property_types: string[];
  average_session_duration: number;
  last_activity: string;
  days_range: number;
}

export interface BehaviorEvent {
  id: string;
  user_id?: string;
  property_id?: number;
  session_id: string;
  event_type: string;
  event_data: Record<string, any>;
  page_url?: string;
  referrer_url?: string;
  device_type?: string;
  browser_name?: string;
  ip_address?: string;
  created_at: string;
}

const getAuthHeaders = () => {
  const token = localStorage.getItem('access_token');
  return token ? { 'Authorization': `Bearer ${token}` } : {};
};

export const behaviorAnalysisApi = {
  async getUserInsights(days?: number): Promise<UserBehaviorInsights> {
    const url = days ? `${API_BASE}/behavior/my-insights?days=${days}` : `${API_BASE}/behavior/my-insights`;
    const response = await fetch(url, {
      headers: getAuthHeaders()
    });
    if (!response.ok) {
      const errorText = await response.text();
      console.error('Failed to fetch insights:', response.status, errorText);
      throw new Error(`Failed to fetch insights: ${response.status}`);
    }
    return response.json();
  },

  async getUserEvents(limit: number = 100, days?: number): Promise<BehaviorEvent[]> {
    const params = new URLSearchParams({ limit: limit.toString() });
    if (days) params.append('days', days.toString());
    const response = await fetch(`${API_BASE}/behavior/my-events?${params}`, {
      headers: getAuthHeaders()
    });
    if (!response.ok) {
      const errorText = await response.text();
      console.error('Events fetch failed:', response.status, errorText);
      throw new Error(`Failed to fetch events: ${response.status}`);
    }
    return response.json();
  },

  async getSpecificUserInsights(userId: string): Promise<UserBehaviorInsights> {
    const response = await fetch(`${API_BASE}/behavior/insights/${userId}`, {
      headers: getAuthHeaders()
    });
    if (!response.ok) throw new Error('Failed to fetch user insights');
    return response.json();
  },

  async getSpecificUserEvents(userId: string, limit: number = 100): Promise<BehaviorEvent[]> {
    const response = await fetch(`${API_BASE}/behavior/events/${userId}?limit=${limit}`, {
      headers: getAuthHeaders()
    });
    if (!response.ok) throw new Error('Failed to fetch user events');
    return response.json();
  }
};
