import axios from 'axios';

// Create axios instance with base URL
const api = axios.create({
  baseURL: 'http://localhost:8090/api',
  headers: {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
  },
  timeout: 10000,
});

// Add request interceptor to include auth token
api.interceptors.request.use((config) => {
  // Check for different token types
  const accessToken = localStorage.getItem('access_token');
  const sellerToken = localStorage.getItem('sellerToken');
  const agentToken = localStorage.getItem('agentToken');
  const legacyToken = localStorage.getItem('token');
  const authToken = localStorage.getItem('auth_token');
  
  const token = accessToken || sellerToken || agentToken || legacyToken || authToken;
  
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});

export interface CampaignData {
  user_id: string;
  campaignName: string;
  campaignObjective: string;
  propertyId: string;
  startDate: string;
  endDate: string;
  budgetType: 'daily' | 'total';
  budgetAmount: number;
  propertyType: string;
  propertyLocation: string;
  propertyPrice: number;
  propertyImages: string[];
  propertyDescription: string;
  primaryCTA: string;
  targetCity: string;
  localityRadius: number;
  budgetRange: string;
  buyerType: string;
  leadChannels: string[];
  agentName: string;
  agentContact: string;
  platforms: string[];
  scheduleType: string;
  billingType: 'prepaid' | 'postpaid';
  paymentMethod: string;
  mediaUsageConsent: boolean;
  dataAccuracyConsent: boolean;
}

export interface CampaignResponse {
  message: string;
  campaign_id: string;
  campaign: any;
}

export const createCampaign = async (campaignData: CampaignData): Promise<CampaignResponse> => {
  const response = await api.post('/campaigns/', campaignData);
  return response.data;
};

export const getCampaigns = async (params?: {
  campaign_id?: string;
  skip?: number;
  limit?: number;
  status?: string;
  summary?: boolean;
  admin?: boolean;
}) => {
  const response = await api.get('/campaigns/', { params });
  return response.data;
};

export const getCampaignsByUserId = async (userId: string, params?: {
  skip?: number;
  limit?: number;
  status?: string;
}) => {
  const response = await api.get(`/campaigns/user/${userId}`, { params });
  return response.data;
};

export const updateCampaign = async (campaignId: string, campaignData: Partial<CampaignData>) => {
  const response = await api.put(`/campaigns/?campaign_id=${campaignId}`, campaignData);
  return response.data;
};

export const deleteCampaign = async (campaignId: string) => {
  const response = await api.delete(`/campaigns/?campaign_id=${campaignId}`);
  return response.data;
};

export const approveCampaign = async (campaignId: string) => {
  const response = await api.patch(`/campaigns/${campaignId}/approve`);
  return response.data;
};

export const sendMarketingQuoteEmail = async (email: string) => {
  const response = await api.post('/campaigns/quote-email', { email });
  return response.data;
};
