// Simple API client for making HTTP requests
export const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

interface ApiResponse<T = any> {
  data: T;
  status: number;
  statusText: string;
  headers: Record<string, string>;
}

class ApiClient {
  private baseURL: string;

  constructor(baseURL: string = API_BASE_URL) {
    this.baseURL = baseURL;
  }

  private getAuthHeaders(): Record<string, string> {
    const token = localStorage.getItem('access_token') || localStorage.getItem('token');
    return token ? { 'Authorization': `Bearer ${token}` } : {};
  }

  private async request<T = any>(
    method: string,
    url: string,
    options: RequestInit = {}
  ): Promise<ApiResponse<T>> {
    const fullUrl = url.startsWith('http') ? url : `${this.baseURL}${url}`;

    const config: RequestInit = {
      method,
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
        ...this.getAuthHeaders(),
        ...options.headers,
      },
      ...options,
    };

    try {
      const response = await fetch(fullUrl, config);

      let data: T;
      const contentType = response.headers.get('content-type');

      if (contentType && contentType.includes('application/json')) {
        data = await response.json();
      } else if (options.headers && (options.headers as any)['responseType'] === 'blob') {
        data = await response.blob() as any;
      } else {
        data = await response.text() as any;
      }

      const apiResponse: ApiResponse<T> = {
        data,
        status: response.status,
        statusText: response.statusText,
        headers: Object.fromEntries(response.headers.entries()),
      };

      if (!response.ok) {
        if (response.status === 401) {
          localStorage.removeItem('access_token');
          localStorage.removeItem('token');
          window.location.href = '/login';
        }
        console.error('API Error Response:', {
          status: response.status,
          statusText: response.statusText,
          data: data
        });

        const error = new Error(`HTTP ${response.status}: ${response.statusText}`);
        (error as any).response = { status: response.status, data };
        throw error;
      }

      return apiResponse;
    } catch (error) {
      console.error(`API request failed: ${method} ${fullUrl}`, error);
      throw error;
    }
  }

  async get<T = any>(url: string, config?: RequestInit): Promise<ApiResponse<T>> {
    return this.request<T>('GET', url, config);
  }

  async post<T = any>(url: string, data?: any, config?: RequestInit): Promise<ApiResponse<T>> {
    return this.request<T>('POST', url, {
      ...config,
      body: data ? JSON.stringify(data) : undefined,
    });
  }

  async put<T = any>(url: string, data?: any, config?: RequestInit): Promise<ApiResponse<T>> {
    return this.request<T>('PUT', url, {
      ...config,
      body: data ? JSON.stringify(data) : undefined,
    });
  }

  async delete<T = any>(url: string, config?: RequestInit): Promise<ApiResponse<T>> {
    return this.request<T>('DELETE', url, config);
  }
}

export const api = new ApiClient();