const API_BASE_URL = 'http://localhost:8090/api';

export interface CollaborationRequest {
  agent_email: string;
  commission_split_secondary: number;
  can_edit_listing?: boolean;
  can_respond_inquiries?: boolean;
  can_schedule_visits?: boolean;
  can_view_analytics?: boolean;
  invitation_message?: string;
}

export interface Collaboration {
  id: string;
  property_id: number;
  primary_agent_id: string;
  secondary_agent_id: string;
  primary_agent_name?: string;
  secondary_agent_name?: string;
  secondary_agent_email?: string;
  property_title?: string;
  collaboration_type: string;
  commission_split_primary: number;
  commission_split_secondary: number;
  status: 'pending' | 'active' | 'completed' | 'cancelled' | 'declined';
  can_edit_listing: boolean;
  can_respond_inquiries: boolean;
  can_schedule_visits: boolean;
  can_view_analytics: boolean;
  invitation_message?: string;
  created_at: string;
  accepted_at?: string;
  declined_at?: string;
}

export interface CollaborationListResponse {
  collaborations: Collaboration[];
  total_count: number;
}

const getAuthHeaders = () => {
  const token = localStorage.getItem('agentToken') || 
               localStorage.getItem('sellerToken') || 
               localStorage.getItem('access_token') || 
               localStorage.getItem('token');
  return token ? { 'Authorization': `Bearer ${token}` } : {};
};

export const collaborationApi = {
  // Send collaboration invitation
  inviteAgent: async (propertyId: number, request: CollaborationRequest) => {
    const response = await fetch(`${API_BASE_URL}/collaborations/properties/${propertyId}/collaborate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        ...getAuthHeaders()
      },
      body: JSON.stringify(request)
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.detail || 'Failed to send invitation');
    }

    return response.json();
  },

  // Get my collaboration invitations
  getMyInvitations: async (): Promise<CollaborationListResponse> => {
    const response = await fetch(`${API_BASE_URL}/collaborations/invitations`, {
      headers: {
        'Accept': 'application/json',
        ...getAuthHeaders()
      }
    });

    if (!response.ok) {
      throw new Error('Failed to fetch invitations');
    }

    return response.json();
  },

  // Accept collaboration invitation
  acceptInvitation: async (collaborationId: string) => {
    const response = await fetch(`${API_BASE_URL}/collaborations/invitations/${collaborationId}/accept`, {
      method: 'PUT',
      headers: {
        'Content-Type': 'application/json',
        ...getAuthHeaders()
      }
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.detail || 'Failed to accept invitation');
    }

    return response.json();
  },

  // Decline collaboration invitation
  declineInvitation: async (collaborationId: string) => {
    const response = await fetch(`${API_BASE_URL}/collaborations/invitations/${collaborationId}/decline`, {
      method: 'PUT',
      headers: {
        'Content-Type': 'application/json',
        ...getAuthHeaders()
      }
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.detail || 'Failed to decline invitation');
    }

    return response.json();
  },

  // Get property collaborations
  getPropertyCollaborations: async (propertyId: number): Promise<CollaborationListResponse> => {
    const response = await fetch(`${API_BASE_URL}/collaborations/properties/${propertyId}`, {
      headers: {
        'Accept': 'application/json',
        ...getAuthHeaders()
      }
    });

    if (!response.ok) {
      throw new Error('Failed to fetch property collaborations');
    }

    return response.json();
  },

  // Get my active collaborations
  getMyCollaborations: async (): Promise<CollaborationListResponse> => {
    const response = await fetch(`${API_BASE_URL}/collaborations/my-collaborations`, {
      headers: {
        'Accept': 'application/json',
        ...getAuthHeaders()
      }
    });

    if (!response.ok) {
      throw new Error('Failed to fetch collaborations');
    }

    return response.json();
  },

  // Revoke collaboration
  revokeCollaboration: async (collaborationId: string) => {
    const response = await fetch(`${API_BASE_URL}/collaborations/collaborations/${collaborationId}`, {
      method: 'DELETE',
      headers: {
        'Content-Type': 'application/json',
        ...getAuthHeaders()
      }
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.detail || 'Failed to revoke collaboration');
    }

    return response.json();
  }
};