import { api } from './client';

export interface CompetitorAnalyticsRequest {
  property_id?: number;
  city: string;
  locality: string;
  property_type: string;
  listing_type: string;
  property_condition?: string;
  bhk?: number;
  builtup_area_sqft: number;
  price: number;
  listed_date?: string;
}

export interface CompetitorAnalyticsResponse {
  property_id?: number;
  city: string;
  locality: string;
  seller_price_per_sqft: number;
  locality_median_price_per_sqft: number;
  locality_average_price_per_sqft: number;
  price_range_per_sqft: {
    min: number;
    max: number;
  };
  difference_percentage: number;
  verdict: 'Underpriced' | 'Fairly Priced' | 'Overpriced' | 'insufficient_data';
  confidence_level: 'High' | 'Medium' | 'Low' | 'Insufficient Data';
  comparable_property_count: number;
  average_days_on_market: number;
  seller_days_on_market?: number;
  seller_insight: string;
  fallback_used?: boolean;
  fallback_reason?: string;
}

export const competitorAnalyticsApi = {
  async analyzeProperty(data: CompetitorAnalyticsRequest): Promise<CompetitorAnalyticsResponse> {
    // Make public API call without authentication
    const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
    const response = await fetch(`${API_BASE_URL}/competitor-analytics/analyze`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(data),
    });
    
    if (!response.ok) {
      const error = await response.json().catch(() => ({ message: 'Analysis failed' }));
      throw new Error(error.message || 'Failed to analyze property');
    }
    
    return response.json();
  },

  async healthCheck(): Promise<{ status: string; service: string }> {
    const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
    const response = await fetch(`${API_BASE_URL}/competitor-analytics/health`);
    return response.json();
  }
};