const API_BASE = 'http://localhost:8090/api';

const getHeaders = () => {
  const token = localStorage.getItem('access_token');
  return {
    'Content-Type': 'application/json',
    ...(token && { 'Authorization': `Bearer ${token}` })
  };
};

export const contractsApi = {
  create: async (data: any) => {
    const response = await fetch(`${API_BASE}/contracts/`, {
      method: 'POST',
      headers: getHeaders(),
      body: JSON.stringify(data)
    });
    if (!response.ok) throw new Error('Failed to create contract');
    return response.json();
  },

  list: async (params?: any) => {
    const queryParams = new URLSearchParams(params).toString();
    const response = await fetch(`${API_BASE}/contracts/?${queryParams}`, {
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to fetch contracts');
    return response.json();
  },

  getById: async (id: string) => {
    const response = await fetch(`${API_BASE}/contracts/${id}`, {
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to fetch contract');
    return response.json();
  },

  sign: async (id: string, data: { signer_role: string; signature_data: string; ip_address?: string }) => {
    const response = await fetch(`${API_BASE}/contracts/${id}/sign`, {
      method: 'POST',
      headers: getHeaders(),
      body: JSON.stringify(data)
    });
    if (!response.ok) throw new Error('Failed to sign contract');
    return response.json();
  },

  getSignatures: async (id: string) => {
    const response = await fetch(`${API_BASE}/contracts/${id}/signatures`, {
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to fetch signatures');
    return response.json();
  },

  updateStatus: async (id: string, status: string) => {
    const response = await fetch(`${API_BASE}/contracts/${id}/status?status=${status}`, {
      method: 'PUT',
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to update contract status');
    return response.json();
  },

  // Digital signing workflow
  initiateSigning: async (id: string) => {
    const response = await fetch(`${API_BASE}/contracts/${id}/initiate-signing`, {
      method: 'POST',
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to initiate signing');
    return response.json();
  },

  signDigital: async (id: string, data: { signature_data: string; device_info?: any }) => {
    const response = await fetch(`${API_BASE}/contracts/${id}/sign-digital`, {
      method: 'POST',
      headers: getHeaders(),
      body: JSON.stringify(data)
    });
    if (!response.ok) throw new Error('Failed to sign contract');
    return response.json();
  },

  getSigningStatus: async (id: string) => {
    const response = await fetch(`${API_BASE}/contracts/${id}/signing-status`, {
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to get signing status');
    return response.json();
  },

  downloadStampPaper: async (id: string) => {
    const response = await fetch(`${API_BASE}/contracts/${id}/stamp-paper`, {
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to download stamp paper');
    return response.blob();
  },

  // Get contracts for a specific user
  getUserContracts: async (userId: string) => {
    const response = await fetch(`${API_BASE}/contracts/user/${userId}`, {
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to fetch user contracts');
    return response.json();
  },

  // Alternative endpoint via user dashboard
  getUserContractsFromDashboard: async (userId: string) => {
    const response = await fetch(`${API_BASE}/user-dashboard/contracts?user_id=${userId}`, {
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to fetch user contracts from dashboard');
    return response.json();
  },

  // Test endpoint
  testContracts: async () => {
    const response = await fetch(`${API_BASE}/user-dashboard/contracts/test`, {
      headers: getHeaders()
    });
    if (!response.ok) throw new Error('Failed to test contracts');
    return response.json();
  }
};
