import { 
  DataRetentionPolicy, 
  ErasureRequest, 
  DataExport, 
  UserDataSummary, 
  RetentionStats,
  ApiResponse,
  PaginatedResponse
} from '../types/dataRetention';

const API_BASE_URL = import.meta.env.VITE_API_URL || 'http://localhost:8090/api';

class DataRetentionAPI {
  private static async request<T>(
    endpoint: string, 
    options: RequestInit = {}
  ): Promise<T> {
    const token = localStorage.getItem('access_token');
    
    const response = await fetch(`${API_BASE_URL}${endpoint}`, {
      ...options,
      headers: {
        'Content-Type': 'application/json',
        ...(token && { Authorization: `Bearer ${token}` }),
        ...options.headers,
      },
    });

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.detail || errorData.message || `HTTP ${response.status}`);
    }

    return response.json();
  }

  // Data Retention Policies
  static async getRetentionPolicies(activeOnly = false): Promise<DataRetentionPolicy[]> {
    const params = activeOnly ? '?active_only=true' : '';
    return this.request<DataRetentionPolicy[]>(`/data-retention/policies${params}`);
  }

  static async createRetentionPolicy(policy: Partial<DataRetentionPolicy>): Promise<DataRetentionPolicy> {
    return this.request<DataRetentionPolicy>('/data-retention/policies', {
      method: 'POST',
      body: JSON.stringify(policy),
    });
  }

  static async updateRetentionPolicy(id: number, policy: Partial<DataRetentionPolicy>): Promise<DataRetentionPolicy> {
    return this.request<DataRetentionPolicy>(`/data-retention/policies/${id}`, {
      method: 'PUT',
      body: JSON.stringify(policy),
    });
  }

  // Erasure Requests
  static async getErasureRequests(filters: {
    status?: string;
    page?: number;
    page_size?: number;
  } = {}): Promise<PaginatedResponse<ErasureRequest>> {
    const params = new URLSearchParams();
    if (filters.status) params.append('status', filters.status);
    if (filters.page) params.append('page', filters.page.toString());
    if (filters.page_size) params.append('page_size', filters.page_size.toString());
    
    const queryString = params.toString();
    return this.request<PaginatedResponse<ErasureRequest>>(
      `/data-retention/erasure-requests${queryString ? `?${queryString}` : ''}`
    );
  }

  static async createErasureRequest(request: {
    request_type: 'partial' | 'complete';
    reason: string;
    data_categories?: string[];
  }): Promise<ErasureRequest> {
    return this.request<ErasureRequest>('/data-retention/erasure-requests', {
      method: 'POST',
      body: JSON.stringify(request),
    });
  }

  static async updateErasureRequest(id: number, updates: {
    status?: 'pending' | 'approved' | 'rejected' | 'completed';
    notes?: string;
  }): Promise<ErasureRequest> {
    return this.request<ErasureRequest>(`/data-retention/erasure-requests/${id}`, {
      method: 'PUT',
      body: JSON.stringify(updates),
    });
  }

  // Data Exports
  static async getDataExports(): Promise<DataExport[]> {
    return this.request<DataExport[]>('/data-retention/exports');
  }

  static async requestDataExport(exportType: 'personal_data' | 'activity_logs' | 'property_data' | 'complete'): Promise<DataExport> {
    return this.request<DataExport>('/data-retention/exports', {
      method: 'POST',
      body: JSON.stringify({ export_type: exportType }),
    });
  }

  static async downloadDataExport(id: number): Promise<Blob> {
    const token = localStorage.getItem('access_token');
    const response = await fetch(`${API_BASE_URL}/data-retention/exports/${id}/download`, {
      headers: {
        ...(token && { Authorization: `Bearer ${token}` }),
      },
    });

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.detail || errorData.message || 'Download failed');
    }

    return response.blob();
  }

  // User Data Summary
  static async getUserDataSummary(): Promise<UserDataSummary> {
    return this.request<UserDataSummary>('/data-retention/user-data-summary');
  }

  // Statistics
  static async getRetentionStats(): Promise<RetentionStats> {
    return this.request<RetentionStats>('/data-retention/stats');
  }

  // Data Cleanup
  static async runDataCleanup(): Promise<{ deleted_records: number; freed_space_mb: number; policies_processed: number }> {
    return this.request('/data-retention/cleanup', {
      method: 'POST',
    });
  }

  // Consent Management
  static async recordConsent(consent: {
    consent_type: 'data_processing' | 'marketing' | 'analytics' | 'third_party_sharing';
    granted: boolean;
    ip_address?: string;
    user_agent?: string;
  }): Promise<{ success: boolean; message: string }> {
    return this.request('/data-retention/consent', {
      method: 'POST',
      body: JSON.stringify(consent),
    });
  }

  static async getUserConsents(): Promise<{ consents: any[] }> {
    return this.request('/data-retention/consent');
  }
}

export default DataRetentionAPI;