const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8090';
const DIGITAL_VAULT_BASE_URL = `${API_BASE_URL}/api/digital-documents`;

export interface DigitalDocument {
  id: string;
  user_id: string;
  name: string;
  description?: string;
  file_name: string;
  file_type: string;
  file_size: number;
  is_password_protected: boolean;
  password_hint?: string;
  uploaded_at: string;
  updated_at: string;
}

export interface DigitalDocumentListResponse {
  documents: DigitalDocument[];
  total_count: number;
}

export interface DigitalDocumentUploadResponse {
  success: boolean;
  document?: DigitalDocument;
  error?: string;
}

// Create/Upload Digital Document
export const uploadDigitalDocument = async (
  name: string,
  description: string,
  file: File,
  password?: string,
  passwordHint?: string
): Promise<DigitalDocumentUploadResponse> => {
  const token = localStorage.getItem('access_token');

  // Convert file to base64
  const reader = new FileReader();
  const base64Promise = new Promise<string>((resolve, reject) => {
    reader.onload = () => resolve(reader.result as string);
    reader.onerror = () => reject(new Error('Failed to read file'));
    reader.readAsDataURL(file);
  });

  try {
    const fileData = await base64Promise;

    const formData = new FormData();
    formData.append('name', name);
    formData.append('description', description);
    formData.append('file_name', file.name);
    formData.append('file_type', file.type);
    formData.append('file_size', file.size.toString());
    formData.append('file_data', fileData);
    if (password) {
      formData.append('password', password);
    }
    if (passwordHint) {
      formData.append('password_hint', passwordHint);
    }

    const response = await fetch(`${DIGITAL_VAULT_BASE_URL}/`, {
      method: 'POST',
      headers: {
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
      body: formData,
    });

    if (!response.ok) {
      const errorText = await response.text();
      return { success: false, error: `HTTP ${response.status}: ${errorText}` };
    }

    const result = await response.json();
    return { success: true, document: result };

  } catch (error) {
    return {
      success: false,
      error: error instanceof Error ? error.message : 'Upload failed'
    };
  }
};

// Get User's Digital Documents
export const getUserDigitalDocuments = async (): Promise<DigitalDocument[]> => {
  const token = localStorage.getItem('access_token');

  try {
    const response = await fetch(`${DIGITAL_VAULT_BASE_URL}/`, {
      method: 'GET',
      headers: {
        'Accept': 'application/json',
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to fetch documents: ${response.status} - ${errorText}`);
    }

    const result: DigitalDocumentListResponse = await response.json();
    return result.documents || [];

  } catch (error) {
    console.error('Error fetching digital documents:', error);
    throw error;
  }
};

// Verify Document Password
export const verifyDocumentPassword = async (documentId: string, password: string): Promise<{ valid: boolean; message: string }> => {
  const token = localStorage.getItem('access_token');

  try {
    const response = await fetch(`${DIGITAL_VAULT_BASE_URL}/${documentId}/verify-password`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
      body: JSON.stringify({ password }),
    });

    if (!response.ok) {
      const errorText = await response.text();
      return { valid: false, message: `Verification failed: ${errorText}` };
    }

    const result = await response.json();
    return { valid: result.valid, message: result.message };

  } catch (error) {
    console.error('Error verifying password:', error);
    return { valid: false, message: 'Password verification failed' };
  }
};

// Get Specific Digital Document
export const getDigitalDocument = async (documentId: string, password?: string): Promise<DigitalDocument> => {
  const token = localStorage.getItem('access_token');

  try {
    let response;
    
    if (password) {
      // Use POST with form data for password-protected documents
      const formData = new FormData();
      formData.append('password', password);
      
      response = await fetch(`${DIGITAL_VAULT_BASE_URL}/${documentId}`, {
        method: 'POST',
        headers: {
          ...(token && { 'Authorization': `Bearer ${token}` }),
        },
        body: formData,
      });
    } else {
      // Use GET for non-password-protected documents
      response = await fetch(`${DIGITAL_VAULT_BASE_URL}/${documentId}`, {
        method: 'GET',
        headers: {
          'Accept': 'application/json',
          ...(token && { 'Authorization': `Bearer ${token}` }),
        },
      });
    }

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to fetch document: ${response.status} - ${errorText}`);
    }

    const result = await response.json();
    return result;

  } catch (error) {
    console.error('Error fetching digital document:', error);
    throw error;
  }
};

// Download Digital Document
export const downloadDigitalDocument = async (documentId: string, password?: string): Promise<void> => {
  const token = localStorage.getItem('access_token');

  try {
    console.log('Starting download for document:', documentId);
    console.log('Using URL:', `${DIGITAL_VAULT_BASE_URL}/${documentId}/download`);

    let response;
    
    if (password) {
      // Use POST with form data for password-protected documents
      const formData = new FormData();
      formData.append('password', password);
      
      response = await fetch(`${DIGITAL_VAULT_BASE_URL}/${documentId}/download`, {
        method: 'POST',
        headers: {
          ...(token && { 'Authorization': `Bearer ${token}` }),
        },
        body: formData,
      });
    } else {
      // Use GET for non-password-protected documents
      response = await fetch(`${DIGITAL_VAULT_BASE_URL}/${documentId}/download`, {
        method: 'GET',
        headers: {
          ...(token && { 'Authorization': `Bearer ${token}` }),
        },
      });
    }

    console.log('Download response status:', response.status);
    console.log('Response headers:', Object.fromEntries(response.headers.entries()));

    if (!response.ok) {
      const errorText = await response.text();
      console.error('Download failed with response:', errorText);
      throw new Error(`Failed to download document: ${response.status} - ${errorText}`);
    }

    // Get the filename from Content-Disposition header
    const contentDisposition = response.headers.get('Content-Disposition');
    let filename = 'document.pdf'; // default
    if (contentDisposition) {
      // Improved regex to handle quoted filenames
      const filenameMatch = contentDisposition.match(/filename[^;=\n]*=((['"]).*?\2|[^;\n]*)/);
      if (filenameMatch && filenameMatch[1]) {
        filename = filenameMatch[1].replace(/['"]/g, '');
      }
    }

    console.log('Extracted filename:', filename);

    // Create blob from response
    const blob = await response.blob();
    console.log('Blob created, size:', blob.size, 'type:', blob.type);

    if (blob.size === 0) {
      throw new Error('Downloaded file is empty');
    }

    // Create download link
    const url = window.URL.createObjectURL(blob);
    const link = window.document.createElement('a');
    link.href = url;
    link.download = filename;
    link.style.display = 'none';

    window.document.body.appendChild(link);
    link.click();
    window.document.body.removeChild(link);

    console.log('Download initiated successfully');

    // Clean up the URL
    setTimeout(() => {
      window.URL.revokeObjectURL(url);
    }, 100);

  } catch (error) {
    console.error('Error downloading digital document:', error);
    const errorMessage = error instanceof Error ? error.message : 'Unknown error occurred';
    alert(`Download failed: ${errorMessage}`);
    throw error;
  }
};

// Update Digital Document
export const updateDigitalDocument = async (
  documentId: string,
  updates: { name?: string; description?: string }
): Promise<DigitalDocument> => {
  const token = localStorage.getItem('access_token');

  const formData = new FormData();
  if (updates.name) formData.append('name', updates.name);
  if (updates.description) formData.append('description', updates.description);

  try {
    const response = await fetch(`${DIGITAL_VAULT_BASE_URL}/${documentId}`, {
      method: 'PUT',
      headers: {
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
      body: formData,
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to update document: ${response.status} - ${errorText}`);
    }

    const result = await response.json();
    return result;

  } catch (error) {
    console.error('Error updating digital document:', error);
    throw error;
  }
};

// Delete Digital Document
export const deleteDigitalDocument = async (documentId: string): Promise<{ success: boolean; message?: string }> => {
  const token = localStorage.getItem('access_token');

  try {
    const response = await fetch(`${DIGITAL_VAULT_BASE_URL}/${documentId}`, {
      method: 'DELETE',
      headers: {
        'Accept': 'application/json',
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to delete document: ${response.status} - ${errorText}`);
    }

    const result = await response.json();
    return { success: true, message: result.message };

  } catch (error) {
    console.error('Error deleting digital document:', error);
    return { success: false };
  }
};