const API_BASE_URL = 'http://localhost:8090';

export interface DocumentUploadResponse {
  success: boolean;
  documents?: Array<{
    id: string;
    file_name: string;
    document_type: string;
    file_size: number;
    upload_date: string;
    download_url?: string;
  }>;
  error?: string;
}

export interface PropertyDocument {
  id: number;
  property_id: number;
  doc_type: string;
  file_url: string;
  user_id: string;
  created_at: string;
  created_by: string;
  updated_by: string;
  updated_time: string;
}

// Upload Property Documents
export const uploadPropertyDocuments = async (
  propertyId: string,
  files: File[],
  documentTypes: string[]
): Promise<DocumentUploadResponse> => {
  const user = JSON.parse(localStorage.getItem('user') || '{}');
  const userId = user.id || user.user_id || '353905a3-36d0-427c-b78a-f80b9fdfa02a';
  
  const uploadedDocuments = [];
  
  // Upload each file individually
  for (let i = 0; i < files.length; i++) {
    const file = files[i];
    const docType = documentTypes[i] || 'other';
    
    const formData = new FormData();
    formData.append('file', file);
    formData.append('property_id', propertyId);
    formData.append('user_id', userId);
    formData.append('doc_type', docType);

    try {
      const response = await fetch(`${API_BASE_URL}/api/documents/upload`, {
        method: 'POST',
        body: formData,
      });

      if (!response.ok) {
        const errorText = await response.text();
        return { success: false, error: `HTTP ${response.status}: ${errorText}` };
      }

      const result = await response.json();
      uploadedDocuments.push(result);
    } catch (error) {
      return { 
        success: false, 
        error: error instanceof Error ? error.message : 'Upload failed' 
      };
    }
  }
  
  return { success: true, documents: uploadedDocuments };
};

// Get Property Documents (with ownership verification)
export const getPropertyDocuments = async (propertyId: string): Promise<PropertyDocument[]> => {
  const user = JSON.parse(localStorage.getItem('user') || '{}');
  const userEmail = user.email || localStorage.getItem('user_email');
  
  const url = new URL(`${API_BASE_URL}/api/legal-health/property/${propertyId}/documents`);
  if (userEmail) {
    url.searchParams.append('user_email', userEmail);
  }
  
  console.log('=== GET DOCUMENTS API DEBUG ===');
  console.log('Request URL:', url.toString());
  console.log('Property ID:', propertyId);
  console.log('User Email:', userEmail);
  
  try {
    const response = await fetch(url.toString(), {
      method: 'GET',
      headers: {
        'Accept': 'application/json',
      },
    });

    console.log('Response status:', response.status);
    console.log('Response ok:', response.ok);

    if (!response.ok) {
      const errorText = await response.text();
      console.log('Error response text:', errorText);
      
      if (response.status === 403) {
        throw new Error('You can only view documents for your own properties');
      }
      
      throw new Error(`Failed to fetch documents: ${response.status} - ${errorText}`);
    }

    const result = await response.json();
    console.log('API Response:', result);
    console.log('Documents in response:', result.documents);
    console.log('===============================');
    
    return result.documents || [];
  } catch (error) {
    console.error('Error fetching property documents:', error);
    throw error;
  }
};

// Generate Download URL for Document (with ownership verification)
export const generateDocumentDownloadURL = async (documentId: string) => {
  const token = localStorage.getItem('access_token');
  const user = JSON.parse(localStorage.getItem('user') || '{}');
  const userEmail = user.email || localStorage.getItem('user_email');
  
  const url = new URL(`${API_BASE_URL}/api/documents/${documentId}/download`);
  if (userEmail) {
    url.searchParams.append('user_email', userEmail);
  }
  
  try {
    const response = await fetch(url.toString(), {
      method: 'GET',
      headers: {
        'Accept': 'application/json',
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
    });

    if (!response.ok) {
      if (response.status === 403) {
        throw new Error('You can only download documents for your own properties');
      }
      throw new Error(`Failed to generate download URL: ${response.status}`);
    }

    const result = await response.json();
    return { success: true, download_url: result.download_url };
  } catch (error) {
    console.error('Error generating download URL:', error);
    return { success: false, error: error instanceof Error ? error.message : 'Failed to generate URL' };
  }
};

// Replace Document
export const replaceDocument = async (documentId: string, newFile: File, documentType?: string) => {
  const token = localStorage.getItem('access_token');
  const formData = new FormData();
  
  formData.append('file', newFile);
  if (documentType) {
    formData.append('document_type', documentType);
  }
  
  try {
    const response = await fetch(`${API_BASE_URL}/api/documents/${documentId}`, {
      method: 'PUT',
      headers: {
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
      body: formData,
    });

    if (!response.ok) {
      const errorText = await response.text();
      return { success: false, error: errorText };
    }

    const result = await response.json();
    return { success: true, ...result };
  } catch (error) {
    return { 
      success: false, 
      error: error instanceof Error ? error.message : 'Replace failed' 
    };
  }
};

// Delete Document
export const deleteDocument = async (documentId: string) => {
  const token = localStorage.getItem('access_token');
  
  try {
    const response = await fetch(`${API_BASE_URL}/api/documents/${documentId}`, {
      method: 'DELETE',
      headers: {
        'Accept': 'application/json',
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to delete document: ${response.status}`);
    }

    return { success: true };
  } catch (error) {
    console.error('Error deleting document:', error);
    return { success: false, error: error instanceof Error ? error.message : 'Delete failed' };
  }
};

// Clear all documents for a property (useful for testing)
export const clearPropertyDocuments = async (propertyId: string) => {
  try {
    const documents = await getPropertyDocuments(propertyId);
    const deletePromises = documents.map(doc => deleteDocument(doc.id.toString()));
    await Promise.all(deletePromises);
    return { success: true };
  } catch (error) {
    return { success: false, error: 'Failed to clear documents' };
  }
};