const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';
const BASE_URL = API_BASE_URL.replace('/api', ''); // Remove /api for endpoints that need base URL

export interface InfraProject {
  id: string;
  project_name: string;
  project_type: string;
  location_lat?: number;
  location_lng?: number;
  city?: string;
  locality?: string;
  impact_radius_km?: number;
  expected_completion_date?: string;
  status: string;
  construction_progress: number;
  impact_score?: number;
}

export interface PriceForecast {
  locality: string;
  city: string;
  property_type: string;
  forecast_period: string;
  current_avg_price?: number;
  forecasted_price?: number;
  appreciation_percentage?: number;
  confidence_score?: number;
  infra_impact_score?: number;
  forecast_factors?: Record<string, any>;
}

export interface Hotspot {
  locality: string;
  city: string;
  hotspot_score?: number;
  growth_potential: string;
  upcoming_projects_count: number;
  infra_density_score?: number;
  demand_trend: string;
  investment_readiness: string;
  key_drivers?: string[];
  risk_factors?: string[];
  recommended_for?: string[];
}

export interface SavedInsight {
  id: string;
  insight_type: string;
  reference_id?: string;
  locality?: string;
  city?: string;
  notes?: string;
  created_at: string;
}

class FutureInsightsService {
  private getAuthHeaders() {
    const token = localStorage.getItem('access_token');
    return {
      'Authorization': `Bearer ${token}`,
      'Content-Type': 'application/json'
    };
  }

  async fetchInfraProjects(filters: {
    city?: string;
    locality?: string;
    project_type?: string;
    status?: string;
  } = {}): Promise<InfraProject[]> {
    const params = new URLSearchParams();
    Object.entries(filters).forEach(([key, value]) => {
      if (value) params.append(key, value);
    });

    const response = await fetch(`${BASE_URL}/api/future-insights/projects?${params}`, {
      headers: this.getAuthHeaders()
    });

    if (!response.ok) {
      throw new Error('Failed to fetch infrastructure projects');
    }

    return response.json();
  }

  async fetchPriceForecasts(filters: {
    city?: string;
    locality?: string;
    property_type?: string;
    forecast_period?: string;
  } = {}): Promise<PriceForecast[]> {
    const params = new URLSearchParams();
    Object.entries(filters).forEach(([key, value]) => {
      if (value) params.append(key, value);
    });

    const response = await fetch(`${BASE_URL}/api/future-insights/forecasts/price?${params}`, {
      headers: this.getAuthHeaders()
    });

    if (!response.ok) {
      throw new Error('Failed to fetch price forecasts');
    }

    return response.json();
  }

  async fetchHotspots(filters: {
    city?: string;
    growth_potential?: string;
    min_score?: number;
    user_role?: string;
  } = {}): Promise<Hotspot[]> {
    const params = new URLSearchParams();
    Object.entries(filters).forEach(([key, value]) => {
      if (value !== undefined) params.append(key, value.toString());
    });

    const response = await fetch(`${BASE_URL}/api/future-insights/hotspots?${params}`, {
      headers: this.getAuthHeaders()
    });

    if (!response.ok) {
      throw new Error('Failed to fetch hotspots');
    }

    return response.json();
  }

  async saveInsight(data: {
    insight_type: string;
    reference_id?: string;
    locality?: string;
    city?: string;
    notes?: string;
  }): Promise<{ success: boolean; insight_id: string }> {
    const response = await fetch(`${BASE_URL}/api/future-insights/saved`, {
      method: 'POST',
      headers: this.getAuthHeaders(),
      body: JSON.stringify(data)
    });

    if (!response.ok) {
      throw new Error('Failed to save insight');
    }

    return response.json();
  }

  async fetchSavedInsights(): Promise<SavedInsight[]> {
    const response = await fetch(`${BASE_URL}/api/future-insights/saved`, {
      headers: this.getAuthHeaders()
    });

    if (!response.ok) {
      throw new Error('Failed to fetch saved insights');
    }

    return response.json();
  }

  async deleteSavedInsight(insightId: string): Promise<{ success: boolean }> {
    const response = await fetch(`${BASE_URL}/api/future-insights/saved/${insightId}`, {
      method: 'DELETE',
      headers: this.getAuthHeaders()
    });

    if (!response.ok) {
      throw new Error('Failed to delete insight');
    }

    return response.json();
  }
}

export const futureInsightsService = new FutureInsightsService();