// Hyperlocal Indicators API
export interface MetroLine {
  name: string;
  distanceMeters: number;
  expectedYear: number;
}

export interface Infrastructure {
  name: string;
  distanceMeters: number;
  expectedYear: number;
}

export interface Zoning {
  name: string;
  distanceMeters: number;
  description: string;
}

export interface HyperlocalIndicators {
  metroLines: MetroLine[];
  infrastructure: Infrastructure[];
  zoning: Zoning[];
}

export const fetchHyperlocalIndicators = async (propertyId: string, radius: number = 5000): Promise<HyperlocalIndicators> => {
  try {
    const response = await fetch(`http://localhost:8090/api/hyperlocal/indicators?propertyId=${propertyId}&radius=${radius}`, {
      method: 'GET',
      headers: { 'Accept': 'application/json' },
      mode: 'cors',
      credentials: 'omit',
    });

    if (response.ok) {
      const data = await response.json();
      
      // Map backend field names to frontend field names
      const mapIndicator = (item: any) => ({
        ...item,
        distanceMeters: item.distance_meters || item.distanceMeters,
        expectedYear: item.expected_year || item.expectedYear
      });
      
      return {
        metroLines: (data.metroLines || data.metro || []).map(mapIndicator),
        infrastructure: (data.infrastructure || []).map(mapIndicator),
        zoning: (data.zoning || []).map(mapIndicator)
      };
    }
    throw new Error(`API responded with status: ${response.status}`);
  } catch (error) {
    console.error('Error fetching hyperlocal indicators:', error);
    return { metroLines: [], infrastructure: [], zoning: [] };
  }
};

// Legacy function for backward compatibility
export const fetchHyperlocalIndicatorsByPropertyId = async (propertyId: string): Promise<HyperlocalIndicators> => {
  // This would need to fetch property coordinates first, then call the main function
  // For now, return empty data
  return { metroLines: [], infrastructure: [], zoning: [] };
};