// Mock API for Hyperlocal Economic Indicators

export interface MetroProject {
  id: string;
  name: string;
  type: string;
  description: string | null;
  latitude: number;
  longitude: number;
  status: 'planned' | 'under_construction' | 'completed' | 'cancelled';
  expected_year: number | null;
  radius_meters: number;
  createdAt: string;
  updatedAt: string | null;
}

export interface InfrastructureProject {
  id: string;
  name: string;
  type: string;
  description: string | null;
  latitude: number;
  longitude: number;
  status: 'planned' | 'under_construction' | 'completed' | 'cancelled';
  expected_year: number | null;
  radius_meters: number;
  createdAt: string;
  updatedAt: string | null;
}

export interface ZoningChange {
  id: string;
  name: string;
  type: string;
  description: string | null;
  latitude: number;
  longitude: number;
  status: 'planned' | 'under_construction' | 'completed' | 'cancelled';
  expected_year: number | null;
  radius_meters: number;
  createdAt: string;
  updatedAt: string | null;
}

const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

const getAuthHeaders = () => {
  const token = localStorage.getItem('access_token');
  return {
    'Content-Type': 'application/json',
    'Accept': 'application/json',
    ...(token && { 'Authorization': `Bearer ${token}` })
  };
};

// Metro Projects API
export const getMetroProjects = async (page = 1, limit = 10, search = '', status = 'all'): Promise<{
  data: MetroProject[];
  total: number;
  page: number;
  limit: number;
}> => {
  console.log('🚇 API: getMetroProjects called', { page, limit, search, status });
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/metro-projects`, {
      method: 'GET',
      headers: getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }
    
    const result = await response.json();
    console.log('🚇 API: Backend response:', result);
    
    return {
      data: result || [],
      total: result.length || 0,
      page: page,
      limit: limit
    };
  } catch (error) {
    console.error('🚇 API Error:', error);
    return { data: [], total: 0, page, limit };
  }
};

export const createMetroProject = async (project: Omit<MetroProject, 'id' | 'createdAt' | 'updatedAt'>): Promise<MetroProject> => {
  console.log('🚇 API: createMetroProject called', project);
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/metro-projects`, {
      method: 'POST',
      headers: getAuthHeaders(),
      body: JSON.stringify(project)
    });
    
    if (!response.ok) {
      const errorText = await response.text();
      console.error('🚇 API Error Response:', errorText);
      throw new Error(`HTTP ${response.status}: ${response.statusText} - ${errorText}`);
    }
    
    const result = await response.json();
    console.log('🚇 API: Project created:', result);
    return result;
  } catch (error) {
    console.error('🚇 Create Error:', error);
    throw error;
  }
};

export const updateMetroProject = async (id: string, updates: Partial<MetroProject>): Promise<MetroProject> => {
  console.log('🚇 API: updateMetroProject called', { id, updates });
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/metro-projects/${id}`, {
      method: 'PUT',
      headers: getAuthHeaders(),
      body: JSON.stringify(updates)
    });
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }
    
    const result = await response.json();
    console.log('🚇 API: Project updated:', result);
    return result;
  } catch (error) {
    console.error('🚇 Update Error:', error);
    throw error;
  }
};

export const deleteMetroProject = async (id: string): Promise<void> => {
  console.log('🚇 API: deleteMetroProject called', { id });
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/metro-projects/${id}`, {
      method: 'DELETE',
      headers: getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }
    
    console.log('🚇 API: Project deleted successfully');
  } catch (error) {
    console.error('🚇 Delete Error:', error);
    throw error;
  }
};

// Infrastructure Projects API
export const getInfrastructureProjects = async (page = 1, limit = 10, search = '', type = 'all', status = 'all'): Promise<{
  data: InfrastructureProject[];
  total: number;
  page: number;
  limit: number;
}> => {
  console.log('🏢 API: getInfrastructureProjects called', { page, limit, search, type, status });
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/infrastructure-projects`, {
      method: 'GET',
      headers: getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }
    
    const result = await response.json();
    console.log('🏢 API: Backend response:', result);
    
    return {
      data: result || [],
      total: result.length || 0,
      page: page,
      limit: limit
    };
  } catch (error) {
    console.error('🏢 API Error:', error);
    return { data: [], total: 0, page, limit };
  }
};

export const createInfrastructureProject = async (project: Omit<InfrastructureProject, 'id' | 'createdAt' | 'updatedAt'>): Promise<InfrastructureProject> => {
  console.log('🏢 API: createInfrastructureProject called', project);
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/infrastructure-projects`, {
      method: 'POST',
      headers: getAuthHeaders(),
      body: JSON.stringify(project)
    });
    
    if (!response.ok) {
      const errorText = await response.text();
      console.error('🏢 API Error Response:', errorText);
      throw new Error(`HTTP ${response.status}: ${response.statusText} - ${errorText}`);
    }
    
    const result = await response.json();
    console.log('🏢 API: Project created:', result);
    return result;
  } catch (error) {
    console.error('🏢 Create Error:', error);
    throw error;
  }
};

export const updateInfrastructureProject = async (id: string, updates: Partial<InfrastructureProject>): Promise<InfrastructureProject> => {
  console.log('🏢 API: updateInfrastructureProject called', { id, updates });
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/infrastructure-projects/${id}`, {
      method: 'PUT',
      headers: getAuthHeaders(),
      body: JSON.stringify(updates)
    });
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }
    
    const result = await response.json();
    console.log('🏢 API: Project updated:', result);
    return result;
  } catch (error) {
    console.error('🏢 Update Error:', error);
    throw error;
  }
};

export const deleteInfrastructureProject = async (id: string): Promise<void> => {
  console.log('🏢 API: deleteInfrastructureProject called', { id });
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/infrastructure-projects/${id}`, {
      method: 'DELETE',
      headers: getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }
    
    console.log('🏢 API: Project deleted successfully');
  } catch (error) {
    console.error('🏢 Delete Error:', error);
    throw error;
  }
};

// Zoning Changes API
export const getZoningChanges = async (page = 1, limit = 10, search = '', city = 'all'): Promise<{
  data: ZoningChange[];
  total: number;
  page: number;
  limit: number;
}> => {
  console.log('🗺️ API: getZoningChanges called', { page, limit, search, city });
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/zoning-changes`, {
      method: 'GET',
      headers: getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }
    
    const result = await response.json();
    console.log('🗺️ API: Backend response:', result);
    
    return {
      data: result || [],
      total: result.length || 0,
      page: page,
      limit: limit
    };
  } catch (error) {
    console.error('🗺️ API Error:', error);
    return { data: [], total: 0, page, limit };
  }
};

export const createZoningChange = async (zoning: Omit<ZoningChange, 'id' | 'createdAt' | 'updatedAt'>): Promise<ZoningChange> => {
  console.log('🗺️ API: createZoningChange called', zoning);
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/zoning-changes`, {
      method: 'POST',
      headers: getAuthHeaders(),
      body: JSON.stringify(zoning)
    });
    
    if (!response.ok) {
      const errorText = await response.text();
      console.error('🗺️ API Error Response:', errorText);
      throw new Error(`HTTP ${response.status}: ${response.statusText} - ${errorText}`);
    }
    
    const result = await response.json();
    console.log('🗺️ API: Zoning change created:', result);
    return result;
  } catch (error) {
    console.error('🗺️ Create Error:', error);
    throw error;
  }
};

export const updateZoningChange = async (id: string, updates: Partial<ZoningChange>): Promise<ZoningChange> => {
  console.log('🗺️ API: updateZoningChange called', { id, updates });
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/zoning-changes/${id}`, {
      method: 'PUT',
      headers: getAuthHeaders(),
      body: JSON.stringify(updates)
    });
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }
    
    const result = await response.json();
    console.log('🗺️ API: Zoning change updated:', result);
    return result;
  } catch (error) {
    console.error('🗺️ Update Error:', error);
    throw error;
  }
};

export const deleteZoningChange = async (id: string): Promise<void> => {
  console.log('🗺️ API: deleteZoningChange called', { id });
  
  try {
    const response = await fetch(`${API_BASE_URL}/hyperlocal/zoning-changes/${id}`, {
      method: 'DELETE',
      headers: getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`HTTP ${response.status}: ${response.statusText}`);
    }
    
    console.log('🗺️ API: Zoning change deleted successfully');
  } catch (error) {
    console.error('🗺️ Delete Error:', error);
    throw error;
  }
};