import { API_BASE_URL } from './client';

export interface ImpersonationResponse {
  success: boolean;
  message: string;
  role: string;
  access_token: string;
  refresh_token: string;
  token_type: string;
  expires_in: number;
}

/**
 * Simulate a role for admin (Admin acts as Buyer/Seller/Agent)
 */
export const simulateRole = async (role: 'buyer' | 'seller' | 'agent'): Promise<ImpersonationResponse> => {
  const token = localStorage.getItem('access_token');

  if (!token) {
    throw new Error('No authentication token found');
  }

  const response = await fetch(`${API_BASE_URL}/admin/impersonate/role/${role}`, {
    method: 'POST',
    headers: {
      'Authorization': `Bearer ${token}`,
      'Content-Type': 'application/json'
    }
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.detail || 'Failed to simulate role');
  }

  return response.json();
};

/**
 * Check if currently impersonating
 */
export const isImpersonating = (): boolean => {
  return localStorage.getItem('admin_restore_token') !== null;
};

/**
 * Get current impersonation info from token
 */
export const getImpersonationInfo = (): { role: string; originalRole: string } | null => {
  const token = localStorage.getItem('access_token');
  if (!token) return null;

  try {
    const payload = JSON.parse(atob(token.split('.')[1]));
    if (payload.is_impersonating) {
      const roleMap: { [key: number]: string } = {
        2: 'Admin',
        1: 'Buyer',
        3: 'Seller',
        4: 'Agent'
      };
      return {
        role: roleMap[payload.role_id] || 'Unknown',
        originalRole: roleMap[payload.original_role_id] || 'Unknown'
      };
    }
  } catch (e) {
    console.error('Failed to parse token:', e);
  }

  return null;
};

/**
 * Restore admin session
 */
export const restoreAdminSession = (): void => {
  const adminToken = localStorage.getItem('admin_restore_token');

  if (adminToken) {
    localStorage.setItem('access_token', adminToken);
    localStorage.removeItem('admin_restore_token');
    window.location.href = '/admin/dashboard';
  }
};
