const API_BASE = import.meta.env.VITE_API_BASE || '/api';

export interface InquiryRequest {
  property_id: number;
  reason_to_buy: string;
  buying_timeline: 'Immediate' | 'Within 1 month' | 'Within 3 months' | 'Within 6 months' | 'More than 6 months';
  interested_in_loan: boolean;
  interested_in_visit: boolean;
  buyer_name: string;
  buyer_email: string;
  buyer_phone: string;
  // Rental specific fields
  preferred_move_in?: string;
  rental_duration?: string;
  enquiry_for?: 'SALE' | 'RENTAL';
}

export interface Inquiry {
  inquiry_id: string;
  buyer_id: string;
  owner_id: string;
  property_id: number;
  reason_to_buy: string;
  buying_timeline: string;
  interested_in_loan: boolean;
  interested_in_visit: boolean;
  status: string;
  created_at: string;
  property_title: string;
  property_address: string;
  property_price: number;
  buyer_name: string;
  buyer_email: string;
  buyer_phone: string;
  owner_name?: string;
  owner_email: string;
  owner_phone: string;
}

export interface InquiriesResponse {
  inquiries: Inquiry[];
  total_count: number;
  page: number;
  limit: number;
}

// Create inquiry - FIXED to use correct endpoint
export const createInquiry = async (userId: string, inquiryData: InquiryRequest): Promise<{ message: string; inquiry_id: string }> => {
  const token = localStorage.getItem('access_token');
  
  console.log('📝 Creating inquiry:', {
    userId,
    inquiryData,
    endpoint: `${API_BASE}/user-dashboard/inquiries?user_id=${userId}`
  });
  
  const response = await fetch(`${API_BASE}/user-dashboard/inquiries?user_id=${userId}`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      ...(token && { 'Authorization': `Bearer ${token}` })
    },
    body: JSON.stringify(inquiryData)
  });

  console.log('📝 Inquiry API response status:', response.status);

  if (!response.ok) {
    const errorText = await response.text();
    console.error('📝 Inquiry API error:', response.status, errorText);
    try {
      const error = JSON.parse(errorText);
      throw new Error(error.detail || error.message || 'Failed to create inquiry');
    } catch (parseError) {
      throw new Error(`API Error ${response.status}: ${errorText}`);
    }
  }

  const result = await response.json();
  console.log('📝 Inquiry created successfully:', result);
  
  // Also create lead directly to ensure it appears in agent dashboard
  try {
    console.log('📝 Creating lead directly...');
    const leadResponse = await fetch(`${API_BASE}/leads-management/leads`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        ...(token && { 'Authorization': `Bearer ${token}` })
      },
      body: JSON.stringify({
        property_id: inquiryData.property_id,
        name: inquiryData.buyer_name,
        email: inquiryData.buyer_email,
        phone: inquiryData.buyer_phone,
        message: inquiryData.reason_to_buy,
        budget: null,
        preferred_contact: 'call',
        lead_type: 'interested',
        target_role: 'agent'
      })
    });
    
    if (leadResponse.ok) {
      const leadResult = await leadResponse.json();
      console.log('📝 Lead created successfully:', leadResult);
    } else {
      console.warn('📝 Lead creation failed, but inquiry was successful');
    }
  } catch (leadError) {
    console.warn('📝 Lead creation error:', leadError);
  }
  
  return result;
};

// Get sent inquiries (buyer) - FIXED to use correct endpoint
export const getSentInquiries = async (
  userId: string,
  page: number = 1,
  limit: number = 20
): Promise<InquiriesResponse> => {
  const token = localStorage.getItem('access_token');
  
  const params = new URLSearchParams({
    user_id: userId,
    page: page.toString(),
    limit: limit.toString()
  });

  const response = await fetch(`${API_BASE}/user-dashboard/inquiries?${params}`, {
    headers: {
      ...(token && { 'Authorization': `Bearer ${token}` })
    }
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.detail || error.message || 'Failed to fetch sent inquiries');
  }

  return response.json();
};

// Get received inquiries (seller) - FIXED to use correct endpoint
export const getReceivedInquiries = async (
  sellerId: string,
  page: number = 1,
  limit: number = 20
): Promise<InquiriesResponse> => {
  const token = localStorage.getItem('access_token');
  
  const params = new URLSearchParams({
    page: page.toString(),
    limit: limit.toString()
  });

  const response = await fetch(`${API_BASE}/user-dashboard/${sellerId}/inquiries?${params}`, {
    headers: {
      ...(token && { 'Authorization': `Bearer ${token}` })
    }
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.detail || error.message || 'Failed to fetch received inquiries');
  }

  return response.json();
};

// Check if user has already inquired about a property
export const checkExistingInquiry = async (userId: string, propertyId: number): Promise<boolean> => {
  try {
    const inquiries = await getSentInquiries(userId, 1, 100);
    return inquiries.inquiries.some(inquiry => inquiry.property_id === propertyId);
  } catch (error) {
    console.error('Error checking existing inquiry:', error);
    return false;
  }
};