const API_BASE = import.meta.env.VITE_API_BASE || '/api';

export interface InvestmentInsight {
  propertyId: number;
  investmentPotential: 'High' | 'Medium' | 'Low';
  cagr5Year: number;
  rentalYield: number;
  priceAppreciation: number[];
  demandScore: number;
  localityTrend: 'Rising' | 'Stable' | 'Declining';
  roiProjection: {
    year1: number;
    year3: number;
    year5: number;
  };
}

export interface LocalityInsight {
  locality: string;
  city: string;
  avgPrice: number;
  priceGrowth: number;
  demandLevel: 'High' | 'Medium' | 'Low';
  rentalYield: number;
  futureScore: number;
}

export interface MarketHeatmapData {
  locality: string;
  lat: number;
  lng: number;
  priceLevel: number;
  demandLevel: number;
  roiScore: number;
}

// API Functions
export const getMarketTrends = async (city?: string) => {
  const params = city ? `?city=${encodeURIComponent(city)}` : '';
  const response = await fetch(`${API_BASE}/investment/market-trends/${params}`);
  if (!response.ok) throw new Error('Failed to fetch market trends');
  return response.json();
};

export const getLocalityInsights = async (city?: string) => {
  const params = city ? `?city=${encodeURIComponent(city)}` : '';
  const response = await fetch(`${API_BASE}/investment/locality-insights/${params}`);
  if (!response.ok) throw new Error('Failed to fetch locality insights');
  return response.json();
};

export const getPropertyROI = async (city?: string) => {
  const params = city ? `?city=${encodeURIComponent(city)}` : '';
  const response = await fetch(`${API_BASE}/investment/property-roi/${params}`);
  if (!response.ok) throw new Error('Failed to fetch property ROI');
  return response.json();
};

export const getPropertyInvestmentInsight = async (propertyId: number): Promise<InvestmentInsight> => {
  const response = await fetch(`${API_BASE}/investment/property/${propertyId}`);
  if (!response.ok) throw new Error('Failed to fetch investment insight');
  return response.json();
};

export const getTopLocalitiesForInvestment = async (city: string): Promise<LocalityInsight[]> => {
  const response = await fetch(`${API_BASE}/investment/localities/${city}`);
  if (!response.ok) throw new Error('Failed to fetch locality insights');
  return response.json();
};

export const getMarketHeatmapData = async (city: string): Promise<MarketHeatmapData[]> => {
  const response = await fetch(`${API_BASE}/investment/heatmap/${city}`);
  if (!response.ok) throw new Error('Failed to fetch heatmap data');
  return response.json();
};

export const getInvestmentReport = async (propertyId: number) => {
  const response = await fetch(`${API_BASE}/investment/report/${propertyId}`);
  if (!response.ok) throw new Error('Failed to fetch investment report');
  return response.json();
};

export const getWeeklyMarketMovers = async (city: string) => {
  const response = await fetch(`${API_BASE}/investment/market-movers/${city}`);
  if (!response.ok) throw new Error('Failed to fetch market movers');
  return response.json();
};