// API Contract for Loan Calculator
export interface BankRateRequest {
  state?: string;
  city?: string;
  loanAmount?: number;
  tenure?: number;
}

export interface BankRateResponse {
  success: boolean;
  data: {
    banks: BankRate[];
    lastUpdated: string;
    location: string;
  };
  error?: string;
}

export interface BankRate {
  bankId: string;
  bankName: string;
  logo: string;
  rates: {
    minRate: number; // precision: 2 decimal places
    maxRate: number;
    rateByTenure: Record<number, number>; // tenure in years -> rate
  };
  processingFee: {
    type: 'percentage' | 'fixed';
    value: number;
    gst: boolean;
  };
  features: string[];
  eligibility: {
    minAmount: number;
    maxAmount: number;
    minTenure: number;
    maxTenure: number;
  };
}

// API Client
const API_BASE = import.meta.env.VITE_API_BASE || '/api';

export const loanCalculatorAPI = {
  // GET /api/banks/loan-rates?state=karnataka&city=bangalore
  getBankRates: async (params: BankRateRequest = {}): Promise<BankRateResponse> => {
    const queryParams = new URLSearchParams();
    if (params.state) queryParams.append('state', params.state);
    if (params.city) queryParams.append('city', params.city);
    if (params.loanAmount) queryParams.append('loanAmount', params.loanAmount.toString());
    if (params.tenure) queryParams.append('tenure', params.tenure.toString());

    const response = await fetch(`${API_BASE}/banks/loan-rates?${queryParams}`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      }
    });

    if (!response.ok) {
      throw new Error(`API Error: ${response.status} - ${response.statusText}`);
    }

    return response.json();
  },

  // POST /api/banks/calculate-emi
  calculateEMI: async (data: {
    bankId: string;
    loanAmount: number;
    tenure: number; // in years
    interestRate?: number;
  }) => {
    const response = await fetch(`${API_BASE}/banks/calculate-emi`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(data)
    });

    if (!response.ok) {
      throw new Error(`EMI Calculation failed: ${response.status}`);
    }

    return response.json();
  },

  // GET /api/banks/health - Health check
  healthCheck: async () => {
    const response = await fetch(`${API_BASE}/banks/health`);
    return response.ok;
  }
};