import { getUserId, validateUserAuthentication, getSessionUserId } from '../utils/userUtils';

const API_BASE_URL = 'http://localhost:8090';

export interface MediaUploadResponse {
  success: boolean;
  images?: Array<{
    id: string;
    url: string;
    category: string;
    is_primary: boolean;
  }>;
  videos?: Array<{
    id: string;
    url: string;
    thumbnail_url?: string;
    title: string;
  }>;
  documents?: Array<{
    id: string;
    url: string;
    document_type: string;
    file_name: string;
  }>;
  error?: string;
}

// Upload property images using correct API with parallel processing
export const uploadPropertyImages = async (
  propertyId: string,
  images: File[]
): Promise<MediaUploadResponse> => {
  const token = localStorage.getItem('access_token');
  
  // Fix 2: Use session-based user ID that persists across property and media creation
  const userId = getSessionUserId();
  
  const uploadedImages = [];
  const BATCH_SIZE = 3; // Upload 3 files simultaneously
  
  try {
    // Process images in batches of 3 for parallel upload
    for (let i = 0; i < images.length; i += BATCH_SIZE) {
      const batch = images.slice(i, i + BATCH_SIZE);
      
      const batchPromises = batch.map(async (image) => {
        const formData = new FormData();
        formData.append('file', image);
        formData.append('property_id', propertyId);
        formData.append('user_id', userId);
        formData.append('media_type', 'image');

        const response = await fetch(`${API_BASE_URL}/api/property-media/upload`, {
          method: 'POST',
          headers: {
            ...(token && { 'Authorization': `Bearer ${token}` }),
          },
          body: formData,
        });

        if (!response.ok) {
          const errorText = await response.text();
          console.error('Image upload failed:', errorText);
          return null;
        }

        const result = await response.json();
        return {
          id: result.id,
          url: result.file_url,
          category: 'general',
          is_primary: result.is_primary || false
        };
      });

      const batchResults = await Promise.all(batchPromises);
      uploadedImages.push(...batchResults.filter(result => result !== null));
    }

    return { success: true, images: uploadedImages };
  } catch (error) {
    return { 
      success: false, 
      error: error instanceof Error ? error.message : 'Upload failed' 
    };
  }
};

// Upload property videos using correct API with parallel processing
export const uploadPropertyVideos = async (
  propertyId: string,
  videos: File[]
): Promise<MediaUploadResponse> => {
  const token = localStorage.getItem('access_token');
  
  // Fix 2: Use session-based user ID that persists across property and media creation
  const userId = getSessionUserId();
  
  const uploadedVideos = [];
  const BATCH_SIZE = 2; // Upload 2 videos simultaneously (videos are larger)
  
  try {
    // Process videos in batches of 2 for parallel upload
    for (let i = 0; i < videos.length; i += BATCH_SIZE) {
      const batch = videos.slice(i, i + BATCH_SIZE);
      
      const batchPromises = batch.map(async (video) => {
        const formData = new FormData();
        formData.append('file', video);
        formData.append('property_id', propertyId);
        formData.append('user_id', userId);
        formData.append('media_type', 'video');

        const response = await fetch(`${API_BASE_URL}/api/property-media/upload`, {
          method: 'POST',
          headers: {
            ...(token && { 'Authorization': `Bearer ${token}` }),
          },
          body: formData,
        });

        if (!response.ok) {
          const errorText = await response.text();
          console.error('Video upload failed:', errorText);
          return null;
        }

        const result = await response.json();
        return {
          id: result.id,
          url: result.file_url,
          thumbnail_url: result.thumbnail_url,
          title: video.name
        };
      });

      const batchResults = await Promise.all(batchPromises);
      uploadedVideos.push(...batchResults.filter(result => result !== null));
    }

    return { success: true, videos: uploadedVideos };
  } catch (error) {
    return { 
      success: false, 
      error: error instanceof Error ? error.message : 'Upload failed' 
    };
  }
};

// Upload property documents
export const uploadPropertyDocuments = async (
  propertyId: string,
  documents: File[],
  documentTypes: string[]
): Promise<MediaUploadResponse> => {
  const token = localStorage.getItem('access_token');
  const formData = new FormData();
  
  documents.forEach((doc, index) => {
    formData.append('files', doc);
    formData.append('document_type', documentTypes[index] || 'general');
  });

  try {
    const response = await fetch(`${API_BASE_URL}/api/property-media/${propertyId}/documents`, {
      method: 'POST',
      headers: {
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
      body: formData,
    });

    if (!response.ok) {
      const errorText = await response.text();
      return { success: false, error: errorText };
    }

    const result = await response.json();
    return { success: true, documents: result.documents || result };
  } catch (error) {
    return { 
      success: false, 
      error: error instanceof Error ? error.message : 'Upload failed' 
    };
  }
};

// Get property media using correct API
export const getPropertyMedia = async (propertyId: string) => {
  // Check all possible token locations for different dashboard types
  const accessToken = localStorage.getItem('access_token');
  const token = localStorage.getItem('token');
  const sellerToken = localStorage.getItem('sellerToken');
  const buyerToken = localStorage.getItem('buyerToken');
  const agentToken = localStorage.getItem('agentToken');
  const userToken = localStorage.getItem('userToken');
  
  // Use the first available token
  const authToken = accessToken || token || sellerToken || buyerToken || agentToken || userToken;
  
  console.log('🔍 Fetching media for property:', propertyId);
  console.log('🔑 localStorage contents:', {
    access_token: localStorage.getItem('access_token') ? 'Present' : 'Missing',
    token: localStorage.getItem('token') ? 'Present' : 'Missing', 
    sellerToken: localStorage.getItem('sellerToken') ? 'Present' : 'Missing',
    buyerToken: localStorage.getItem('buyerToken') ? 'Present' : 'Missing',
    agentToken: localStorage.getItem('agentToken') ? 'Present' : 'Missing',
    userToken: localStorage.getItem('userToken') ? 'Present' : 'Missing'
  });
  console.log('🔑 Using token:', authToken ? 'Present' : 'Missing');
  
  try {
    const response = await fetch(`${API_BASE_URL}/api/property-media/${propertyId}`, {
      method: 'GET',
      headers: {
        'Accept': 'application/json',
        ...(authToken && { 'Authorization': `Bearer ${authToken}` }),
      },
    });

    console.log('📡 API Response status:', response.status);
    console.log('📡 API Response headers:', response.headers);

    if (!response.ok) {
      console.error('❌ API Error:', response.status, response.statusText);
      throw new Error(`Failed to fetch media: ${response.status}`);
    }

    const data = await response.json();
    console.log('📦 Raw API response:', data);
    return data;
  } catch (error) {
    console.error('💥 Error fetching property media:', error);
    return { images: [], videos: [], documents: [] };
  }
};

// Generate download URL for media
export const generateDownloadURL = async (mediaId: string) => {
  // Check all possible token locations for different dashboard types
  const accessToken = localStorage.getItem('access_token');
  const token = localStorage.getItem('token');
  const sellerToken = localStorage.getItem('sellerToken');
  const buyerToken = localStorage.getItem('buyerToken');
  const agentToken = localStorage.getItem('agentToken');
  const userToken = localStorage.getItem('userToken');
  
  // Use the first available token
  const authToken = accessToken || token || sellerToken || buyerToken || agentToken || userToken;
  
  try {
    const response = await fetch(`${API_BASE_URL}/api/property-media/${mediaId}/download`, {
      method: 'GET',
      headers: {
        'Accept': 'application/json',
        ...(authToken && { 'Authorization': `Bearer ${authToken}` }),
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to generate download URL: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error generating download URL:', error);
    return { success: false, error: error instanceof Error ? error.message : 'Failed to generate URL' };
  }
};

// Replace existing media
export const replaceMedia = async (mediaId: string, newFile: File) => {
  const token = localStorage.getItem('access_token');
  const formData = new FormData();
  formData.append('file', newFile);
  
  try {
    const response = await fetch(`${API_BASE_URL}/api/property-media/${mediaId}`, {
      method: 'PUT',
      headers: {
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
      body: formData,
    });

    if (!response.ok) {
      const errorText = await response.text();
      return { success: false, error: errorText };
    }

    const result = await response.json();
    return { success: true, ...result };
  } catch (error) {
    return { 
      success: false, 
      error: error instanceof Error ? error.message : 'Replace failed' 
    };
  }
};

// Delete media file
export const deleteMedia = async (mediaId: string) => {
  const token = localStorage.getItem('access_token');
  
  try {
    const response = await fetch(`${API_BASE_URL}/api/property-media/${mediaId}`, {
      method: 'DELETE',
      headers: {
        'Accept': 'application/json',
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to delete media: ${response.status}`);
    }

    return { success: true };
  } catch (error) {
    console.error('Error deleting media:', error);
    return { success: false, error: error instanceof Error ? error.message : 'Delete failed' };
  }
};

// Get media metadata
export const getMediaMetadata = async (mediaId: string) => {
  const token = localStorage.getItem('access_token');
  
  try {
    const response = await fetch(`${API_BASE_URL}/api/property-media/${mediaId}/metadata`, {
      method: 'GET',
      headers: {
        'Accept': 'application/json',
        ...(token && { 'Authorization': `Bearer ${token}` }),
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch metadata: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error fetching media metadata:', error);
    return { success: false, error: error instanceof Error ? error.message : 'Failed to fetch metadata' };
  }
};