import { InfraProject, PriceForecast, Hotspot } from './futureInsights';

// Mock data for demonstration
const mockProjects: InfraProject[] = [
  {
    id: '1',
    project_name: 'Mumbai Metro Line 3 (Colaba-Bandra-SEEPZ)',
    project_type: 'metro_rail',
    city: 'Mumbai',
    locality: 'Bandra',
    status: 'in_progress',
    construction_progress: 75,
    expected_completion_date: '2024-12-31',
    impact_score: 9,
    impact_radius_km: 5
  },
  {
    id: '2',
    project_name: 'Coastal Road Project',
    project_type: 'highway',
    city: 'Mumbai',
    locality: 'Marine Drive',
    status: 'in_progress',
    construction_progress: 60,
    expected_completion_date: '2025-06-30',
    impact_score: 8,
    impact_radius_km: 3
  },
  {
    id: '3',
    project_name: 'Bangalore Metro Purple Line Extension',
    project_type: 'metro_rail',
    city: 'Bangalore',
    locality: 'Whitefield',
    status: 'approved',
    construction_progress: 25,
    expected_completion_date: '2026-03-31',
    impact_score: 8,
    impact_radius_km: 4
  }
];

const mockForecasts: PriceForecast[] = [
  {
    locality: 'Bandra',
    city: 'Mumbai',
    property_type: 'apartment',
    forecast_period: '1_year',
    current_avg_price: 25000000,
    forecasted_price: 28000000,
    appreciation_percentage: 12,
    confidence_score: 0.85,
    infra_impact_score: 9
  },
  {
    locality: 'Whitefield',
    city: 'Bangalore',
    property_type: 'apartment',
    forecast_period: '1_year',
    current_avg_price: 8000000,
    forecasted_price: 9200000,
    appreciation_percentage: 15,
    confidence_score: 0.78,
    infra_impact_score: 8
  }
];

const mockHotspots: Hotspot[] = [
  {
    locality: 'Bandra West',
    city: 'Mumbai',
    hotspot_score: 9.2,
    growth_potential: 'high',
    upcoming_projects_count: 3,
    infra_density_score: 8.5,
    demand_trend: 'increasing',
    investment_readiness: 'ready',
    key_drivers: ['Metro connectivity', 'Commercial hub', 'Premium location'],
    risk_factors: ['High property prices', 'Traffic congestion'],
    recommended_for: ['Long-term investment', 'Premium buyers']
  },
  {
    locality: 'Whitefield',
    city: 'Bangalore',
    hotspot_score: 8.7,
    growth_potential: 'high',
    upcoming_projects_count: 2,
    infra_density_score: 7.8,
    demand_trend: 'stable',
    investment_readiness: 'ready',
    key_drivers: ['IT hub', 'Metro extension', 'Airport connectivity'],
    risk_factors: ['Market saturation', 'Regulatory changes'],
    recommended_for: ['IT professionals', 'Rental investment']
  }
];

export class MockFutureInsightsService {
  async fetchInfraProjects(filters: {
    city?: string;
    locality?: string;
    project_type?: string;
    status?: string;
  } = {}): Promise<InfraProject[]> {
    // Simulate API delay
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    let filtered = mockProjects;
    
    if (filters.city) {
      filtered = filtered.filter(p => 
        p.city?.toLowerCase().includes(filters.city!.toLowerCase())
      );
    }
    
    if (filters.locality) {
      filtered = filtered.filter(p => 
        p.locality?.toLowerCase().includes(filters.locality!.toLowerCase())
      );
    }
    
    return filtered;
  }

  async fetchPriceForecasts(filters: {
    city?: string;
    locality?: string;
    property_type?: string;
    forecast_period?: string;
  } = {}): Promise<PriceForecast[]> {
    await new Promise(resolve => setTimeout(resolve, 800));
    
    let filtered = mockForecasts;
    
    if (filters.city) {
      filtered = filtered.filter(f => 
        f.city.toLowerCase().includes(filters.city!.toLowerCase())
      );
    }
    
    if (filters.locality) {
      filtered = filtered.filter(f => 
        f.locality.toLowerCase().includes(filters.locality!.toLowerCase())
      );
    }
    
    return filtered;
  }

  async fetchHotspots(filters: {
    city?: string;
    growth_potential?: string;
    min_score?: number;
    user_role?: string;
  } = {}): Promise<Hotspot[]> {
    await new Promise(resolve => setTimeout(resolve, 600));
    
    let filtered = mockHotspots;
    
    if (filters.city) {
      filtered = filtered.filter(h => 
        h.city.toLowerCase().includes(filters.city!.toLowerCase())
      );
    }
    
    if (filters.min_score) {
      filtered = filtered.filter(h => 
        (h.hotspot_score || 0) >= filters.min_score!
      );
    }
    
    return filtered;
  }
}

export const mockFutureInsightsService = new MockFutureInsightsService();