import axios, { AxiosResponse } from 'axios';
import {
  CreateNegotiationRequest,
  SubmitOfferRequest,
  RejectOfferRequest,
  NegotiationResponse,
  NegotiationTimeline
} from '../types/negotiation';

const API_BASE_URL = import.meta.env.VITE_API_URL || 'http://localhost:8090/api';

// Create axios instance with JWT token
const apiClient = axios.create({
  baseURL: API_BASE_URL,
  headers: {
    'Content-Type': 'application/json',
  },
});

// Add JWT token to requests
apiClient.interceptors.request.use((config) => {
  const token = localStorage.getItem('access_token') || 
                sessionStorage.getItem('access_token') ||
                localStorage.getItem('sellerToken') ||
                sessionStorage.getItem('sellerToken') ||
                localStorage.getItem('agentToken') ||
                sessionStorage.getItem('agentToken');
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});

// Handle API errors
apiClient.interceptors.response.use(
  (response) => response,
  (error) => {
    if (error.response?.status === 401) {
      // Token expired or invalid
      localStorage.removeItem('access_token');
      sessionStorage.removeItem('access_token');
      localStorage.removeItem('sellerToken');
      sessionStorage.removeItem('sellerToken');
      localStorage.removeItem('agentToken');
      sessionStorage.removeItem('agentToken');
      window.location.href = '/login';
    }
    return Promise.reject(error);
  }
);

export const negotiationApi = {
  // List all negotiations for current user
  listNegotiations: async (): Promise<any[]> => {
    const response: AxiosResponse<any[]> = await apiClient.get('/negotiations');
    return response.data;
  },

  // Clear all negotiations for current user
  clearAllNegotiations: async (): Promise<void> => {
    await apiClient.delete('/negotiations/clear-all');
  },

  // Create new negotiation
  createNegotiation: async (data: CreateNegotiationRequest): Promise<NegotiationResponse> => {
    const response: AxiosResponse<NegotiationResponse> = await apiClient.post('/negotiations', data);
    return response.data;
  },

  // Submit offer or counter-offer
  submitOffer: async (negotiationId: string, data: SubmitOfferRequest): Promise<NegotiationResponse> => {
    const response: AxiosResponse<NegotiationResponse> = await apiClient.post(
      `/negotiations/${negotiationId}/offer`,
      data
    );
    return response.data;
  },

  // Accept offer
  acceptOffer: async (eventId: string): Promise<NegotiationResponse> => {
    const response: AxiosResponse<NegotiationResponse> = await apiClient.post(
      `/negotiations/${eventId}/accept`,
      {}
    );
    return response.data;
  },

  // Reject offer
  rejectOffer: async (eventId: string, data?: RejectOfferRequest): Promise<NegotiationResponse> => {
    const response: AxiosResponse<NegotiationResponse> = await apiClient.post(
      `/negotiations/${eventId}/reject`,
      data || {}
    );
    return response.data;
  },

  // Get negotiation timeline
  getTimeline: async (negotiationId: string): Promise<NegotiationTimeline> => {
    const response: AxiosResponse<NegotiationTimeline> = await apiClient.get(
      `/negotiations/${negotiationId}/timeline`
    );
    return response.data;
  },
};