const API_BASE_URL = 'http://localhost:8090';

export interface Notification {
  id: string;
  type: 'property' | 'visit' | 'user' | 'document' | 'property_approval' | 'visit_approval' | 'risk_approved' | 'risk_rejected' | 'admin_action';
  title: string;
  message: string;
  time: string;
  read: boolean;
  data?: {
    propertyId?: string;
    property_id?: string;
    visitId?: string;
    visit_id?: string;
    userId?: string;
    user_id?: string;
    documentId?: string;
    document_id?: string;
    propertyName?: string;
    property_title?: string;
    propertyTitle?: string;
    count?: number;
    status?: string;
    reason?: string;
    submission_id?: string;
    action?: string;
  };
}

export interface NotificationResponse {
  notifications: Notification[];
  total_count: number;
  unread_count: number;
  admin_confirmations_count?: number;
}

// Generic API request function
const apiRequest = async (endpoint: string, options: RequestInit = {}) => {
  const token = localStorage.getItem('access_token');
  const response = await fetch(`${API_BASE_URL}${endpoint}`, {
    headers: {
      'Content-Type': 'application/json',
      ...(token && { Authorization: `Bearer ${token}` }),
      ...options.headers,
    },
    ...options,
  });

  if (!response.ok) {
    throw new Error(`API Error: ${response.status}`);
  }

  return response.json();
};

// Get admin notifications
export const getAdminNotifications = async (): Promise<NotificationResponse> => {
  try {
    // Get admin user ID from localStorage
    const adminUser = JSON.parse(localStorage.getItem('user') || '{}');
    const adminId = adminUser.user_id || adminUser.id;
    
    if (!adminId) {
      throw new Error('Admin ID not found');
    }
    
    const response = await apiRequest(`/api/admin/notifications?admin_id=${adminId}`);
    
    // Transform the response to match our notification interface
    const transformedNotifications = response.notifications.map((notif: any) => ({
      id: notif.id,
      type: notif.type,
      title: decodeHtmlEntities(notif.title || ''),
      message: decodeHtmlEntities(notif.message || ''),
      time: notif.created_at || new Date().toISOString(),
      read: notif.read || false,
      data: notif.data || {}
    }));
    
    return {
      notifications: transformedNotifications,
      total_count: response.total_count,
      unread_count: response.unread_count,
      admin_confirmations_count: response.admin_confirmations_count || 0
    };
  } catch (error) {
    console.error('Error fetching notifications:', error);
    // Return mock data with real pending counts
    try {
      const [pendingProperties, pendingVisits] = await Promise.all([
        import('../api/adminProperties').then(api => api.getPendingProperties({ page: 1, limit: 10 })),
        import('../api/adminVisitSchedules').then(api => api.getAdminVisitSchedules(1, 1, 'pending'))
      ]);

      const notifications: Notification[] = [];
      
      if (pendingProperties.total_count > 0 && pendingProperties.properties.length > 0) {
        // Use the most recent property's creation time
        const latestProperty = pendingProperties.properties[0];
        notifications.push({
          id: 'property-pending',
          type: 'property_approval',
          title: 'Property Approvals Pending',
          message: `${pendingProperties.total_count} properties require approval`,
          time: latestProperty.created_at,
          read: false,
          data: { count: pendingProperties.total_count, type: 'property_approval' }
        });
      }

      if (pendingVisits.total_count > 0 && pendingVisits.visits.length > 0) {
        // Use the most recent visit request's creation time
        const latestVisit = pendingVisits.visits[0];
        notifications.push({
          id: 'visit-pending',
          type: 'visit_approval',
          title: 'Visit Schedule Requests',
          message: `${pendingVisits.total_count} visit requests pending approval`,
          time: latestVisit.created_at,
          read: false,
          data: { count: pendingVisits.total_count, type: 'visit_approval' }
        });
      }

      return {
        notifications,
        total_count: notifications.length,
        unread_count: notifications.filter(n => !n.read).length
      };
    } catch (fallbackError) {
      console.error('Error fetching fallback data:', fallbackError);
      return {
        notifications: [],
        total_count: 0,
        unread_count: 0
      };
    }
  }
};

// Mark notification as read
export const markNotificationAsRead = async (notificationId: string): Promise<void> => {
  try {
    // Get admin user ID from localStorage
    const adminUser = JSON.parse(localStorage.getItem('user') || '{}');
    const adminId = adminUser.user_id || adminUser.id;
    
    if (!adminId) {
      throw new Error('Admin ID not found');
    }
    
    await apiRequest(`/api/admin/notifications/${notificationId}/read?admin_id=${adminId}`, {
      method: 'POST'
    });
  } catch (error) {
    console.error('Error marking notification as read:', error);
  }
};

// Mark all notifications as read
export const markAllNotificationsAsRead = async (): Promise<void> => {
  try {
    // Get admin user ID from localStorage
    const adminUser = JSON.parse(localStorage.getItem('user') || '{}');
    const adminId = adminUser.user_id || adminUser.id;
    
    if (!adminId) {
      throw new Error('Admin ID not found');
    }
    
    await apiRequest(`/api/admin/notifications/read-all?admin_id=${adminId}`, {
      method: 'POST'
    });
  } catch (error) {
    console.error('Error marking all notifications as read:', error);
  }
};

// Delete admin notification
export const deleteAdminNotification = async (notificationId: string): Promise<void> => {
  try {
    const adminUser = JSON.parse(localStorage.getItem('user') || '{}');
    const adminId = adminUser.user_id || adminUser.id;
    
    if (!adminId) {
      throw new Error('Admin ID not found');
    }
    
    await apiRequest(`/api/admin/notifications/${notificationId}?admin_id=${adminId}`, {
      method: 'DELETE'
    });
  } catch (error) {
    console.error('Error deleting admin notification:', error);
  }
};

// Delete all admin notifications
export const deleteAllAdminNotifications = async (): Promise<void> => {
  try {
    const adminUser = JSON.parse(localStorage.getItem('user') || '{}');
    const adminId = adminUser.user_id || adminUser.id;
    
    if (!adminId) {
      throw new Error('Admin ID not found');
    }
    
    await apiRequest(`/api/admin/notifications?admin_id=${adminId}`, {
      method: 'DELETE'
    });
  } catch (error) {
    console.error('Error deleting all admin notifications:', error);
  }
};

// Get user notifications
export const getUserNotifications = async (userId: string): Promise<NotificationResponse> => {
  if (!userId || userId === 'user') {
    return {
      notifications: [],
      total_count: 0,
      unread_count: 0
    };
  }

  try {
    const apiResult = await apiRequest(`/api/user-dashboard/notifications?user_id=${userId}`);
    if (apiResult && apiResult.notifications) {
      return apiResult;
    }
    throw new Error('Empty API response');
  } catch (error) {
    console.error('Error fetching user notifications:', error);
    return {
      notifications: [],
      total_count: 0,
      unread_count: 0
    };
  }
};

// Helper function to decode HTML entities
const decodeHtmlEntities = (text: string): string => {
  if (!text) return '';
  
  // Create a temporary element to decode HTML entities
  const textarea = document.createElement('textarea');
  textarea.innerHTML = text;
  const decoded = textarea.value;
  
  // Additional manual decoding for common entities that might not be caught
  return decoded
    .replace(/&#39;/g, "'")
    .replace(/&quot;/g, '"')
    .replace(/&amp;/g, '&')
    .replace(/&lt;/g, '<')
    .replace(/&gt;/g, '>')
    .replace(/&nbsp;/g, ' ');
};

// Mark user notification as read
export const markUserNotificationAsRead = async (notificationId: string): Promise<void> => {
  try {
    const userId = JSON.parse(localStorage.getItem('user') || '{}').user_id || 
                   JSON.parse(localStorage.getItem('user') || '{}').id;
    if (!userId) {
      throw new Error('User ID not found');
    }
    await apiRequest(`/api/user-dashboard/notifications/${notificationId}/read?user_id=${userId}`, {
      method: 'PUT'
    });
  } catch (error) {
    console.error('Error marking user notification as read:', error);
  }
};

// Mark all user notifications as read
export const markAllUserNotificationsAsRead = async (userId: string): Promise<void> => {
  try {
    await apiRequest(`/api/user-dashboard/notifications/mark-all-read?user_id=${userId}`, {
      method: 'PUT'
    });
  } catch (error) {
    console.error('Error marking all user notifications as read:', error);
  }
};

// Delete notification
export const deleteNotification = async (notificationId: string): Promise<void> => {
  try {
    const userId = JSON.parse(localStorage.getItem('user') || '{}').user_id || 
                   JSON.parse(localStorage.getItem('user') || '{}').id;
    if (!userId) {
      throw new Error('User ID not found');
    }
    await apiRequest(`/api/user-dashboard/notifications/${notificationId}?user_id=${userId}`, {
      method: 'DELETE'
    });
  } catch (error) {
    console.error('Error deleting notification:', error);
  }
};

// Delete all user notifications
export const deleteAllUserNotifications = async (userId: string): Promise<void> => {
  try {
    await apiRequest(`/api/user-dashboard/notifications?user_id=${userId}`, {
      method: 'DELETE'
    });
  } catch (error) {
    console.error('Error deleting all user notifications:', error);
  }
};