const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export interface NRIAssistanceRequest {
  fullName: string;
  email: string;
  phone: string;
  countryOfResidence: string;
  assistanceCategory: string;
  priorityLevel: string;
  description: string;
  propertyId: string;
  propertyArea: string;
  propertyCity: string;
}

export interface NRIRequest {
  id: string;
  full_name: string;
  email: string;
  phone: string;
  country: string;
  category: string;
  priority_level: string;
  description: string;
  status: string;
  created_at: string;
}

export const createNRIAssistanceRequest = async (data: NRIAssistanceRequest) => {
  // Map frontend field names to backend expected field names
  const apiData = {
    full_name: data.fullName,
    email: data.email,
    phone: data.phone,
    country: data.countryOfResidence,
    category: data.assistanceCategory,
    priority_level: data.priorityLevel,
    description: data.description,
    property_id: data.propertyId,
    area_locality: data.propertyArea,
    city: data.propertyCity
  };

  const response = await fetch(`${API_BASE_URL}/nri-assistance/requests`, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'Accept': 'application/json',
    },
    body: JSON.stringify(apiData),
    mode: 'cors',
    credentials: 'omit'
  });

  if (!response.ok) {
    const errorText = await response.text();
    throw new Error(`Failed to submit request: ${response.status} ${errorText}`);
  }

  return response.json();
};

export const getNRIAssistanceRequests = async (): Promise<NRIRequest[]> => {
  const response = await fetch(`${API_BASE_URL}/nri-assistance/requests`, {
    method: 'GET',
    headers: {
      'Accept': 'application/json',
    },
    mode: 'cors',
    credentials: 'omit'
  });

  if (!response.ok) {
    throw new Error(`Failed to fetch requests: ${response.status}`);
  }

  const data = await response.json();
  return data.requests || data || [];
};

export const updateNRIRequestStatus = async (requestId: string, status: 'approved' | 'rejected') => {
  const endpoint = status === 'approved' 
    ? `${API_BASE_URL}/nri-assistance/requests/${requestId}/approve`
    : `${API_BASE_URL}/nri-assistance/requests/${requestId}/reject`;

  const response = await fetch(endpoint, {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json',
      'Accept': 'application/json',
    },
    body: JSON.stringify({}),
    mode: 'cors',
    credentials: 'omit'
  });

  if (!response.ok) {
    const errorText = await response.text();
    throw new Error(`Failed to ${status === 'approved' ? 'approve' : 'reject'} request: ${response.status} ${errorText}`);
  }

  return response.json();
};