import axios from 'axios';

const API_BASE_URL = 'http://localhost:8090/api/admin/platform';

export interface PlatformConfig {
  id: number;
  config_key: string;
  config_value: string;
  config_type: 'string' | 'number' | 'boolean' | 'json';
  category: 'general' | 'business' | 'email' | 'payment' | 'features';
  description?: string;
  is_sensitive: boolean;
  updated_by?: string;
  created_at: string;
  updated_at: string;
}

export interface ConfigAuditLog {
  id: number;
  config_key: string;
  old_value?: string;
  new_value: string;
  changed_by: string;
  changed_at: string;
  ip_address?: string;
}

const getAuthHeaders = () => {
  const token = localStorage.getItem('access_token');
  return {
    headers: {
      Authorization: `Bearer ${token}`,
      'Content-Type': 'application/json',
    },
  };
};

export const platformConfigAPI = {
  // Get all configurations
  getAllConfigs: async (category?: string): Promise<{ configs: PlatformConfig[]; total_count: number }> => {
    const url = category ? `${API_BASE_URL}/configs?category=${category}` : `${API_BASE_URL}/configs`;
    const response = await axios.get(url, getAuthHeaders());
    return response.data;
  },

  // Get single configuration
  getConfig: async (configKey: string): Promise<PlatformConfig> => {
    const response = await axios.get(`${API_BASE_URL}/configs/${configKey}`, getAuthHeaders());
    return response.data;
  },

  // Update configuration
  updateConfig: async (configKey: string, configValue: string): Promise<PlatformConfig> => {
    const response = await axios.put(
      `${API_BASE_URL}/configs/${configKey}`,
      { config_value: configValue },
      getAuthHeaders()
    );
    return response.data;
  },

  // Get audit logs
  getAuditLogs: async (configKey?: string, limit: number = 50): Promise<{ logs: ConfigAuditLog[]; total_count: number }> => {
    const url = configKey 
      ? `${API_BASE_URL}/audit/logs?config_key=${configKey}&limit=${limit}`
      : `${API_BASE_URL}/audit/logs?limit=${limit}`;
    const response = await axios.get(url, getAuthHeaders());
    return response.data;
  },

  // Reload configurations
  reloadConfigs: async (): Promise<{ message: string; cached_cleared?: number }> => {
    const response = await axios.post(`${API_BASE_URL}/reload`, {}, getAuthHeaders());
    return response.data;
  },
};
