// Get API base URL from environment or use default
const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

// Check if backend is available
let backendAvailable = true;
const checkBackend = async () => {
  try {
    const response = await fetch(`${API_BASE_URL}/health`, { 
      method: 'GET',
      signal: AbortSignal.timeout(5000)
    });
    backendAvailable = response.ok;
  } catch {
    backendAvailable = false;
  }
};

// Check backend availability on module load
checkBackend();

// Enums matching backend
export enum PropertyType {
  APARTMENT = 'APARTMENT',
  VILLA = 'VILLA',
  HOUSE = 'HOUSE',
  COMMERCIAL = 'COMMERCIAL',
  PLOT = 'PLOT'
}

export enum ListingType {
  SELL = 'SELL',
  RENT = 'RENT',
  LEASE = 'LEASE',
  URGENT_SALE = 'URGENT-SALE',
  BIDDING = 'BIDDING'
}

// Request/Response interfaces
export interface PriceBreakdown {
  base_price: number;
  age_factor: number;
  amenity_bonus: number;
  listing_type_adjustment: number;
  furnishing_bonus: number;
  floor_premium: number;
}

export interface PredictionRequest {
  query?: string;
  property_type: PropertyType;
  listing_type: ListingType;
  bedrooms: number;
  bathrooms: number;
  balconies?: number;
  built_up_area: number;
  carpet_area?: number;
  locality: string;
  furnishing?: string;
  age_years?: number;
  amenities?: string[];
  floor_number?: number;
  total_floors?: number;
  prediction_year?: number;
}

export interface PredictionResponse {
  prediction_id: string;
  predicted_price: number;
  price_range: {
    min: number;
    max: number;
  };
  confidence_score: number;
  price_breakdown: PriceBreakdown;
  rag_context: string[];
  market_insights: string;
  comparable_properties?: any[];
  model_version: string;
  listing_type: string;
  property_type: string;
  locality: string;
  prediction_year: number;
  psf_used?: number;
  created_at: string;
}

export interface MarketInsightsResponse {
  location: string;
  zone?: string;
  avg_psf_2025?: number;
  growth_rate_2025?: number;
  price_trend_2020_2030?: Record<string, number>;
  demand_level?: string;
  supply_status?: string;
  investment_score?: number;
  key_insights: string[];
}

/**
 * Predict property price using AI/ML model
 */
export const predictPrice = async (request: PredictionRequest): Promise<PredictionResponse> => {
  const controller = new AbortController();
  const timeoutId = setTimeout(() => controller.abort(), 30000); // 30 second timeout

  try {
    const response = await fetch(`${API_BASE_URL}/price-prediction/predict`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      },
      body: JSON.stringify(request),
      signal: controller.signal
    });

    clearTimeout(timeoutId);

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.detail || `Prediction failed: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    clearTimeout(timeoutId);
    if (error.name === 'AbortError') {
      throw new Error('Request timeout - Please check if the backend server is running');
    }
    console.error('Error predicting price:', error);
    throw new Error(error.message || 'Failed to connect to prediction service');
  }
};

/**
 * Get comprehensive PSF forecasting for a location
 */
export const getPsfForecasting = async (location: string): Promise<MarketInsightsResponse> => {
  const controller = new AbortController();
  const timeoutId = setTimeout(() => controller.abort(), 15000); // 15 second timeout

  try {
    const queryParams = new URLSearchParams({ location });
    const response = await fetch(
      `${API_BASE_URL}/price-prediction/psf-forecasting?${queryParams.toString()}`,
      {
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json'
        },
        signal: controller.signal
      }
    );

    clearTimeout(timeoutId);

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));
      throw new Error(errorData.detail || `PSF forecasting failed: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    clearTimeout(timeoutId);
    if (error.name === 'AbortError') {
      throw new Error('Request timeout - Please check if the backend server is running');
    }
    console.error('Error fetching PSF forecasting:', error);
    throw new Error(error.message || 'Failed to connect to forecasting service');
  }
};

// Backward compatibility alias
export const getMarketAnalysis = getPsfForecasting;

/**
 * Format currency for display
 */
export const formatCurrency = (amount: number, listingType?: string): string => {
  const isRent = listingType === 'RENT';
  
  if (amount >= 10000000) {
    return `₹${(amount / 10000000).toFixed(2)} Cr${isRent ? '/month' : ''}`;
  } else if (amount >= 100000) {
    return `₹${(amount / 100000).toFixed(2)} L${isRent ? '/month' : ''}`;
  } else {
    return `₹${amount.toLocaleString('en-IN')}${isRent ? '/month' : ''}`;
  }
};

/**
 * Get confidence color based on score
 */
export const getConfidenceColor = (score: number): string => {
  if (score >= 0.9) return 'text-green-600';
  if (score >= 0.75) return 'text-blue-600';
  if (score >= 0.6) return 'text-yellow-600';
  return 'text-orange-600';
};