import { refreshToken } from './auth';

const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

// Profile API Interfaces
export interface UserProfile {
  user_id: string;
  phone_number: string;
  first_name: string;
  last_name: string;
  email: string;
  date_of_birth: string;
  gender: 'male' | 'female' | 'other';
  country: string;
  city: string;
  address: string;
  postal_code: string;
  profile_picture_url: string;
  bio: string;
  preferences: Record<string, any>;
  profile_completion_status: 'complete' | 'incomplete';
  profile_completion_percentage: number;
  is_verified: boolean;
  status: string;
  created_at: string;
  updated_at: string;
  last_login: string;
}

export interface ProfileCompleteRequest {
  profile_data: {
    first_name: string;
    last_name: string;
    email?: string;
    phone_number?: string;
    date_of_birth?: string;
    gender?: 'male' | 'female' | 'other';
    country?: string;
    city?: string;
    address?: string;
    postal_code?: string;
    profile_picture_url?: string;
    bio?: string;
    preferences?: Record<string, any>;
  };
}

// Get user profile by user_id
export const getUserProfileById = async (userId: string): Promise<UserProfile> => {
  const accessToken = localStorage.getItem('access_token');
  if (!accessToken) {
    throw new Error('No access token');
  }

  try {
    const response = await fetch(`${API_BASE_URL}/profile/${userId}`, {
      method: 'GET',
      headers: {
        'Authorization': `Bearer ${accessToken}`,
        'Content-Type': 'application/json',
      },
    });

    if (response.status === 401) {
      await refreshToken();
      const newAccessToken = localStorage.getItem('access_token');
      const retryResponse = await fetch(`${API_BASE_URL}/profile/${userId}`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${newAccessToken}`,
          'Content-Type': 'application/json',
        },
      });
      
      if (!retryResponse.ok) {
        throw new Error('Failed to get user profile');
      }
      return await retryResponse.json();
    }

    if (!response.ok) {
      throw new Error('Failed to get user profile');
    }

    return await response.json();
  } catch (error: any) {
    console.error('User profile fetch error:', error);
    throw error;
  }
};

// Get current user's profile
export const getUserProfile = async (): Promise<UserProfile> => {
  const accessToken = localStorage.getItem('access_token');
  if (!accessToken) {
    throw new Error('No access token');
  }

  try {
    const response = await fetch(`${API_BASE_URL}/profile/me`, {
      method: 'GET',
      headers: {
        'Authorization': `Bearer ${accessToken}`,
        'Content-Type': 'application/json',
      },
    });

    if (response.status === 401) {
      await refreshToken();
      const newAccessToken = localStorage.getItem('access_token');
      const retryResponse = await fetch(`${API_BASE_URL}/profile/me`, {
        method: 'GET',
        headers: {
          'Authorization': `Bearer ${newAccessToken}`,
          'Content-Type': 'application/json',
        },
      });
      
      if (!retryResponse.ok) {
        throw new Error('Failed to get profile');
      }
      return await retryResponse.json();
    }

    if (!response.ok) {
      // Check if response is JSON before parsing
      const contentType = response.headers.get('content-type');
      if (contentType && contentType.includes('application/json')) {
        const errorData = await response.json().catch(() => ({ message: 'Failed to get profile' }));
        throw new Error(errorData.message || 'Failed to get profile');
      } else {
        throw new Error(`Failed to get profile: ${response.status} ${response.statusText}`);
      }
    }

    const profileData = await response.json();
    
    return profileData;
  } catch (error: any) {
    console.error('Profile fetch error:', error);
    // Return default profile structure if backend fails
    return {
      user_id: localStorage.getItem('user_id') || '',
      phone_number: '',
      first_name: '',
      last_name: '',
      email: '',
      date_of_birth: '',
      gender: 'male',
      country: '',
      city: '',
      address: '',
      postal_code: '',
      profile_picture_url: '',
      bio: '',
      preferences: {},
      profile_completion_status: 'incomplete',
      profile_completion_percentage: 0,
      is_verified: false,
      status: 'active',
      created_at: new Date().toISOString(),
      updated_at: new Date().toISOString(),
      last_login: new Date().toISOString()
    };
  }
};

// Update current user's profile
export const updateUserProfile = async (profileData: Partial<UserProfile>): Promise<UserProfile> => {
  const accessToken = localStorage.getItem('access_token');
  if (!accessToken) {
    throw new Error('No access token');
  }

  try {
    const response = await fetch(`${API_BASE_URL}/profile/me`, {
      method: 'PUT',
      headers: {
        'Authorization': `Bearer ${accessToken}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(profileData),
    });

    if (response.status === 401) {
      await refreshToken();
      const newAccessToken = localStorage.getItem('access_token');
      const retryResponse = await fetch(`${API_BASE_URL}/profile/me`, {
        method: 'PUT',
        headers: {
          'Authorization': `Bearer ${newAccessToken}`,
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(profileData),
      });
      
      if (!retryResponse.ok) {
        throw new Error('Failed to update profile');
      }
      return await retryResponse.json();
    }

    if (!response.ok) {
      // Check if response is JSON before parsing
      const contentType = response.headers.get('content-type');
      if (contentType && contentType.includes('application/json')) {
        const errorData = await response.json().catch(() => ({ message: 'Failed to update profile' }));
        throw new Error(errorData.message || 'Failed to update profile');
      } else {
        throw new Error(`Failed to update profile: ${response.status} ${response.statusText}`);
      }
    }

    return await response.json();
  } catch (error: any) {
    throw error;
  }
};

// Complete new user's profile setup
export const completeProfile = async (profileData: ProfileCompleteRequest): Promise<UserProfile> => {
  const accessToken = localStorage.getItem('access_token');
  if (!accessToken) {
    throw new Error('No access token');
  }

  try {
    const response = await fetch(`${API_BASE_URL}/profile/complete`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${accessToken}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(profileData),
    });

    if (response.status === 401) {
      await refreshToken();
      const newAccessToken = localStorage.getItem('access_token');
      const retryResponse = await fetch(`${API_BASE_URL}/profile/complete`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${newAccessToken}`,
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(profileData),
      });
      
      if (!retryResponse.ok) {
        // Check if response is JSON before parsing
        const contentType = retryResponse.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
          const errorData = await retryResponse.json().catch(() => ({ message: 'Failed to complete profile' }));
          throw new Error(errorData.message || 'Failed to complete profile');
        } else {
          throw new Error(`Failed to complete profile: ${retryResponse.status} ${retryResponse.statusText}`);
        }
      }
      return await retryResponse.json();
    }

    if (!response.ok) {
      // Check if response is JSON before parsing
      const contentType = response.headers.get('content-type');
      if (contentType && contentType.includes('application/json')) {
        const errorData = await response.json().catch(() => ({ message: 'Failed to complete profile' }));
        throw new Error(errorData.message || 'Failed to complete profile');
      } else {
        throw new Error(`Failed to complete profile: ${response.status} ${response.statusText}`);
      }
    }

    return await response.json();
  } catch (error: any) {
    throw error;
  }
};

// Upload profile image
export const uploadProfileImage = async (file: File): Promise<{ message: string; image_url: string; profile: UserProfile }> => {
  const accessToken = localStorage.getItem('access_token');
  if (!accessToken) {
    throw new Error('No access token');
  }

  const formData = new FormData();
  formData.append('file', file);

  try {
    const response = await fetch(`${API_BASE_URL}/profile/upload-image`, {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${accessToken}`,
      },
      body: formData,
    });

    if (response.status === 401) {
      await refreshToken();
      const newAccessToken = localStorage.getItem('access_token');
      const retryResponse = await fetch(`${API_BASE_URL}/profile/upload-image`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${newAccessToken}`,
        },
        body: formData,
      });
      
      if (!retryResponse.ok) {
        throw new Error('Failed to upload image');
      }
      return await retryResponse.json();
    }

    if (!response.ok) {
      throw new Error('Failed to upload image');
    }

    return await response.json();
  } catch (error: any) {
    console.error('Image upload error:', error);
    throw error;
  }
};

// Remove profile image
export const removeProfileImage = async (): Promise<{ message: string; profile: UserProfile }> => {
  const accessToken = localStorage.getItem('access_token');
  if (!accessToken) {
    throw new Error('No access token');
  }

  try {
    const response = await fetch(`${API_BASE_URL}/profile/image`, {
      method: 'DELETE',
      headers: {
        'Authorization': `Bearer ${accessToken}`,
      },
    });

    if (response.status === 401) {
      await refreshToken();
      const newAccessToken = localStorage.getItem('access_token');
      const retryResponse = await fetch(`${API_BASE_URL}/profile/image`, {
        method: 'DELETE',
        headers: {
          'Authorization': `Bearer ${newAccessToken}`,
        },
      });
      
      if (!retryResponse.ok) {
        throw new Error('Failed to remove image');
      }
      return await retryResponse.json();
    }

    if (!response.ok) {
      throw new Error('Failed to remove image');
    }

    return await response.json();
  } catch (error: any) {
    console.error('Image removal error:', error);
    throw error;
  }
};

// Legacy exports for backward compatibility
export { getUserProfile as getProfile };
export { updateUserProfile as updateProfile };