const API_BASE_URL = 'http://localhost:8000';

export interface Promotion {
  id: string;
  property_id: string;
  seller_id: string;
  type: 'featured' | 'premium' | 'spotlight';
  status: 'pending_payment' | 'active' | 'inactive' | 'expired';
  cost: number;
  duration: number;
  boost_factor: number;
  start_date: string;
  end_date: string;
  platforms: string[];
  views: number;
  inquiries: number;
  clicks: number;
  created_at: string;
}

export interface CreatePromotionData {
  property_id: string;
  type: 'featured' | 'premium' | 'spotlight';
  platforms?: string[];
}

export interface PromotionAnalytics {
  views: number;
  inquiries: number;
  clicks: number;
  conversion_rate: number;
  cost_per_view: number;
  cost_per_inquiry: number;
  roi: number;
  daily_stats: Array<{
    date: string;
    views: number;
    inquiries: number;
  }>;
}

export const createPromotion = async (data: CreatePromotionData): Promise<{ promotion: Promotion; payment_required: boolean }> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/promotions/create`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(data)
    });

    if (!response.ok) {
      throw new Error(`Failed to create promotion: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error creating promotion:', error);
    throw error;
  }
};

export const getSellerPromotions = async (sellerId: string): Promise<{ promotions: Promotion[] }> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/promotions/seller/${sellerId}`, {
      headers: {
        'Content-Type': 'application/json',
      }
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch promotions: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error fetching promotions:', error);
    throw error;
  }
};

export const activatePromotion = async (promotionId: string): Promise<{ message: string; status: string }> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/promotions/${promotionId}/activate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      }
    });

    if (!response.ok) {
      throw new Error(`Failed to activate promotion: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error activating promotion:', error);
    throw error;
  }
};

export const getPromotionAnalytics = async (promotionId: string): Promise<PromotionAnalytics> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/promotions/${promotionId}/analytics`, {
      headers: {
        'Content-Type': 'application/json',
      }
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch analytics: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error fetching analytics:', error);
    throw error;
  }
};

export const getPromotionTypes = async (): Promise<{ promotion_types: Record<string, any> }> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/promotions/types`, {
      headers: {
        'Content-Type': 'application/json',
      }
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch promotion types: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error fetching promotion types:', error);
    throw error;
  }
};

export const getFeaturedPromotions = async (): Promise<{ properties: Array<{ property_id: number; type: string; status: string; boost_factor: number }> }> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/promotions/featured`, {
      headers: {
        'Content-Type': 'application/json',
      }
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch featured promotions: ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    console.error('Error fetching featured promotions:', error);
    throw error;
  }
};