const API_BASE_URL = (import.meta as any).env?.VITE_API_BASE?.replace('/api', '') || 'http://localhost:8090';

export interface Property {
  id: number;
  title: string;
  description: string;
  listing_type: string;
  property_type: string;
  status: string;
  address: string;
  city: string;
  locality: string;
  state: string;
  pincode: string;
  bedrooms: number;
  bathrooms: number;
  built_up_area: number;
  carpet_area: number;
  sale_price: number;
  monthly_rent: number;
  urgent_pricing?: number;
  furnishing: string;
  amenities: string[];
  price_negotiable: boolean;
  views: number;
  inquiries: number;
  listed_by_id: string;
  created_at: string;
  updated_at: string;
}

export interface PropertiesResponse {
  properties: Property[];
  total_count: number;
  page: number;
  limit: number;
}

export interface GetPropertiesParams {
  page?: number;
  limit?: number;
  search?: string;
  property_type?: string;
  listing_type?: string;
  city?: string;
  bedrooms?: string;
  bathrooms?: string;
  min_price?: number;
  max_price?: number;
  furnishing?: string;
  monthly_rent?: number;
  loan_available?: boolean;
  amenities?: string;
  ordering?: string;
  commute_workplace?: string;
  max_commute_time?: number;
  transport_mode?: 'driving' | 'transit' | 'walking';
}

export const getProperties = async (params: GetPropertiesParams = {}): Promise<PropertiesResponse> => {
  try {
    const queryParams = new URLSearchParams();
    
    Object.entries(params).forEach(([key, value]) => {
      if (value !== undefined && value !== null && value !== '') {
        queryParams.append(key, value.toString());
      }
    });
    
    const url = `${API_BASE_URL}/api/properties/${queryParams.toString() ? '?' + queryParams.toString() : ''}`;
    const response = await fetch(url, {
      headers: { 'Content-Type': 'application/json' }
    });
    
    if (!response.ok) {
      throw new Error(`Failed to fetch properties: ${response.status}`);
    }
    
    return await response.json();
  } catch (error) {
    console.error('Error fetching properties:', error);
    throw error;
  }
};

export const getPropertyById = async (id: number): Promise<Property> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/properties/${id}`, {
      headers: { 'Content-Type': 'application/json' }
    });
    
    if (!response.ok) {
      throw new Error(`Failed to fetch property: ${response.status}`);
    }
    
    return await response.json();
  } catch (error) {
    console.error('Error fetching property:', error);
    throw error;
  }
};

export const createProperty = async (propertyData: any): Promise<Property> => {
  try {
    const formData = new FormData();
    
    // Add property data as JSON string
    const { images, video, ...propertyFields } = propertyData;
    formData.append('property_data', JSON.stringify(propertyFields));
    
    // Add image files
    if (images && images.length > 0) {
      images.forEach((image: File) => {
        formData.append('images', image);
      });
    }
    
    // Add video file
    if (video) {
      formData.append('video', video);
    }
    
    const response = await fetch(`${API_BASE_URL}/api/properties/`, {
      method: 'POST',
      body: formData  // Don't set Content-Type, let browser set it with boundary
    });
    
    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to create property: ${response.status} - ${errorText}`);
    }
    
    return await response.json();
  } catch (error) {
    console.error('Error creating property:', error);
    throw error;
  }
};

export const updateProperty = async (id: number, propertyData: any): Promise<Property> => {
  try {
    const response = await fetch(`${API_BASE_URL}/api/properties/${id}`, {
      method: 'PUT',
      headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json'
      },
      body: JSON.stringify(propertyData)
    });
    
    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Failed to update property: ${response.status} - ${errorText}`);
    }
    
    return await response.json();
  } catch (error) {
    console.error('Error updating property:', error);
    throw error;
  }
};

export const getRecommendedProperties = async (): Promise<Property[]> => {
  try {
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const token = localStorage.getItem('token');
    const userId = user.id || localStorage.getItem('user_id');
    
    const headers: Record<string, string> = {
      'Content-Type': 'application/json'
    };
    
    if (token) {
      headers['Authorization'] = `Bearer ${token}`;
    }
    
    const queryParams = new URLSearchParams();
    if (userId) {
      queryParams.append('user_id', userId.toString());
    }
    
    const url = `${API_BASE_URL}/api/properties/properties/recommendations${queryParams.toString() ? '?' + queryParams.toString() : ''}`;
    const response = await fetch(url, {
      headers
    });
    
    if (!response.ok) {
      throw new Error(`Failed to fetch recommended properties: ${response.status}`);
    }
    
    const data = await response.json();
    return Array.isArray(data) ? data : data.properties || [];
  } catch (error) {
    console.error('Error fetching recommended properties:', error);
    return [];
  }
};