const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export interface PropertyDocument {
  id: string;
  user_id: string;
  generated_at: string;
  document_data: {
    my_properties: any[];
    wishlist: any[];
    saved_properties: any[];
    search_history: any[];
    scheduled_visits: any[];
    property_visit_requests: any[];
    property_comparisons: any[];
    generated_at: string;
  };
  file_path: string | null;
  status: string;
  created_at: string;
  updated_at: string;
}

export interface AdminPropertyDocument {
  id: string;
  admin_id: string;
  generated_at: string;
  document_data: {
    user_management: any[];
    property_management: any[];
    auction_management: any[];
    inquiry_management: any[];
    visit_schedules: any[];
    document_verifications: any[];
    financials_and_payments: any[];
    campaigns_and_promotions: any[];
    reports_and_analytics: any[];
    content_and_cms: any[];
    security_and_audit: any[];
    property_trends: any[];
    generated_at: string;
  };
  file_path: string | null;
  status: string;
  created_at: string;
  updated_at: string;
}

export const propertyGenerateAPI = {
  async generateDocument(timePeriod: string = '30_days'): Promise<PropertyDocument> {
    const token = localStorage.getItem('access_token');
    const response = await fetch(`${API_BASE}/property-generate/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${token}`
      },
      body: JSON.stringify({ time_period: timePeriod })
    });
    if (!response.ok) throw new Error('Failed to generate document');
    return response.json();
  },

  async getUserDocuments(): Promise<PropertyDocument[]> {
    const token = localStorage.getItem('access_token');
    const response = await fetch(`${API_BASE}/property-generate/`, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    });
    if (!response.ok) throw new Error('Failed to fetch documents');
    return response.json();
  },

  async downloadDocument(documentId: string): Promise<any> {
    const token = localStorage.getItem('access_token');
    const response = await fetch(`${API_BASE}/property-generate/${documentId}/download`, {
      headers: {
        'Authorization': `Bearer ${token}`
      }
    });
    if (!response.ok) throw new Error('Failed to download document');
    return response.json();
  },

  async deleteDocument(documentId: string): Promise<void> {
    console.log('API deleteDocument called with ID:', documentId);

    const token = localStorage.getItem('access_token');
    if (!token) {
      console.error('No authentication token found');
      throw new Error('No authentication token found. Please log in again.');
    }

    console.log('Making DELETE request to:', `${API_BASE}/property-generate/${documentId}`);
    console.log('With token present:', !!token);

    try {
      const response = await fetch(`${API_BASE}/property-generate/${documentId}`, {
        method: 'DELETE',
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });

      console.log('DELETE response status:', response.status);
      console.log('DELETE response ok:', response.ok);

      if (!response.ok) {
        let errorData;
        try {
          errorData = await response.text();
          console.log('Error response data:', errorData);
        } catch (e) {
          errorData = 'Unknown error';
          console.error('Could not read error response:', e);
        }
        throw new Error(`Failed to delete document (${response.status}): ${errorData}`);
      }

      // Try to read response body if any
      try {
        const responseText = await response.text();
        console.log('DELETE response body:', responseText);
      } catch (e) {
        console.log('No response body or could not read it');
      }

      console.log('Document deleted successfully');
    } catch (error) {
      console.error('Network or fetch error in deleteDocument:', error);
      throw error;
    }
  }
};

export const propertyGenerateAdminAPI = {
  async generateAdminDocument(timePeriod: string = '30_days'): Promise<AdminPropertyDocument> {
    // Admin endpoints are excluded from authentication, so no Authorization header needed
    const response = await fetch(`${API_BASE}/property-generate-admin/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({ time_period: timePeriod })
    });
    if (!response.ok) throw new Error('Failed to generate admin document');
    return response.json();
  },

  async getAdminDocuments(): Promise<AdminPropertyDocument[]> {
    // Admin endpoints are excluded from authentication, so no Authorization header needed
    const response = await fetch(`${API_BASE}/property-generate-admin/`, {
      headers: {}
    });
    if (!response.ok) throw new Error('Failed to fetch admin documents');
    return response.json();
  },

  async downloadAdminDocument(documentId: string): Promise<Response> {
    // Admin endpoints are excluded from authentication, so no Authorization header needed
    const response = await fetch(`${API_BASE}/property-generate-admin/${documentId}/download`, {
      headers: {}
    });
    if (!response.ok) throw new Error('Failed to download admin document');
    return response; // Return the response directly for blob handling
  }
};