const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export interface PropertyViewStats {
  property_id: number;
  total_views: number;
  unique_users: number;
  views_24h: number;
  views_7d: number;
  last_viewed?: string;
}

export const propertyViewsAPI = {
  async trackView(propertyId: number, userId?: string): Promise<boolean> {
    try {
      const url = `${API_BASE}/property-views/track?property_id=${propertyId}${userId ? `&user_id=${userId}` : ''}`;
      console.log('🔍 Tracking view:', { propertyId, userId, url });
      
      const response = await fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'User-Agent': navigator.userAgent,
        },
      });
      
      console.log('📊 View tracking response status:', response.status);
      
      if (response.ok) {
        const data = await response.json();
        console.log('📊 View tracking response:', data);
        return data.tracked || true;
      } else {
        console.error('❌ View tracking failed:', response.status, response.statusText);
        const errorText = await response.text();
        console.error('❌ Error details:', errorText);
        return false;
      }
    } catch (error) {
      console.error('Error tracking view:', error);
      return false;
    }
  },

  async getStats(propertyId: number): Promise<PropertyViewStats | null> {
    try {
      const response = await fetch(`${API_BASE}/property-views/stats/${propertyId}`);
      
      if (response.ok) {
        return await response.json();
      }
      return null;
    } catch (error) {
      console.error('Error getting view stats:', error);
      return null;
    }
  },

  async getUserHistory(userId: string, limit = 50) {
    try {
      const url = `${API_BASE}/property-views/user/${userId}/recent`;
      console.log('🔍 Fetching user recent views:', { userId, url, limit });
      
      const response = await fetch(url);
      
      console.log('📊 User recent views response status:', response.status);
      console.log('📊 User recent views response headers:', Object.fromEntries(response.headers.entries()));
      
      if (response.ok) {
        const data = await response.json();
        console.log('📊 User recent views response data:', data);
        console.log('📊 Response data type:', typeof data);
        console.log('📊 Response data keys:', data ? Object.keys(data) : 'null');
        
        // Return the data as-is, let the caller handle the structure
        return data;
      } else {
        console.error('❌ User recent views fetch failed:', response.status, response.statusText);
        const errorText = await response.text();
        console.error('❌ Error details:', errorText);
        return { properties: [], count: 0 };
      }
    } catch (error) {
      console.error('Error getting user recent views:', error);
      return { properties: [], count: 0 };
    }
  }
};