const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export interface PropertyView {
  property_id: number;
  property_title?: string;
  property_location?: string;
  property_price?: string;
  property_image?: string;
  property_type?: string;
  property_bedrooms?: number;
  property_bathrooms?: number;
  property_area?: number;
  property_city?: string;
  property_locality?: string;
  property_latitude?: number;
  property_longitude?: number;
  property_listing_type?: string;
  viewed_at: string;
}

export const propertyViewsFallbackAPI = {
  async trackView(propertyId: number, userId?: string): Promise<boolean> {
    try {
      console.log('🔍 Fallback: Tracking view via user activities:', { propertyId, userId });
      
      if (!userId) {
        console.log('❌ No user ID provided for view tracking');
        return false;
      }

      const response = await fetch(`${API_BASE}/user-activities/`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('access_token')}`,
        },
        body: JSON.stringify({
          user_id: userId,
          activity_type: 'property_view',
          property_id: propertyId,
          activity_data: {
            timestamp: new Date().toISOString(),
            source: 'property_detail_page'
          }
        })
      });

      console.log('📊 Fallback view tracking response status:', response.status);

      if (response.ok) {
        const data = await response.json();
        console.log('📊 Fallback view tracking response:', data);
        return true;
      } else {
        console.error('❌ Fallback view tracking failed:', response.status, response.statusText);
        const errorText = await response.text();
        console.error('❌ Error details:', errorText);
        return false;
      }
    } catch (error) {
      console.error('Error in fallback view tracking:', error);
      return false;
    }
  },

  async getUserHistory(userId: string, limit = 10): Promise<{ views: PropertyView[], count: number }> {
    try {
      console.log('🔍 Fallback: Fetching user history via activities:', { userId, limit });
      
      const response = await fetch(`${API_BASE}/user-activities/?user_id=${userId}&activity_type=property_view&limit=${limit}`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('access_token')}`,
        }
      });

      console.log('📊 Fallback user history response status:', response.status);

      if (response.ok) {
        const data = await response.json();
        console.log('📊 Fallback user history response:', data);
        
        // Convert activities to property views format
        const views: PropertyView[] = (data.results || data.activities || []).map((activity: any) => ({
          property_id: activity.property_id,
          property_title: activity.activity_data?.property_title || `Property ${activity.property_id}`,
          property_location: activity.activity_data?.property_location,
          property_price: activity.activity_data?.property_price,
          property_image: activity.activity_data?.property_image,
          property_type: activity.activity_data?.property_type,
          property_bedrooms: activity.activity_data?.property_bedrooms,
          property_bathrooms: activity.activity_data?.property_bathrooms,
          property_area: activity.activity_data?.property_area,
          property_city: activity.activity_data?.property_city,
          property_locality: activity.activity_data?.property_locality,
          property_latitude: activity.activity_data?.property_latitude,
          property_longitude: activity.activity_data?.property_longitude,
          property_listing_type: activity.activity_data?.property_listing_type,
          viewed_at: activity.created_at || activity.timestamp
        }));

        return {
          views,
          count: views.length
        };
      } else {
        console.error('❌ Fallback user history fetch failed:', response.status, response.statusText);
        return { views: [], count: 0 };
      }
    } catch (error) {
      console.error('Error in fallback user history:', error);
      return { views: [], count: 0 };
    }
  }
};