import axios from 'axios';

const API_BASE_URL = import.meta.env.VITE_API_BASE_URL || 'http://localhost:8000';

export interface LocalityRentalYield {
  city: string;
  locality: string;
  avg_monthly_rent: number;
  avg_property_price: number;
  rental_yield_percentage: number;
  apartment_yield?: number;
  villa_yield?: number;
  sample_size: number;
  last_updated: string;
}

export interface PropertyRentalAnalysis {
  property_id: number;
  expected_monthly_rent: number;
  annual_rental_income: number;
  rental_yield: number;
  demand_score: number;
  occupancy_rate: number;
  net_yield_after_costs: number;
  break_even_years: number;
  locality_avg_yield: number;
}

export interface TopLocality {
  city: string;
  locality: string;
  avg_monthly_rent: number;
  avg_property_price: number;
  rental_yield_percentage: number;
  sample_size: number;
}

export const rentalYieldApi = {
  getLocalityYield: async (city: string, locality: string): Promise<LocalityRentalYield> => {
    const response = await axios.get(
      `${API_BASE_URL}/api/rental-yield/locality/${encodeURIComponent(city)}/${encodeURIComponent(locality)}`
    );
    return response.data;
  },

  getTopLocalities: async (city?: string, limit: number = 10): Promise<TopLocality[]> => {
    const params = new URLSearchParams();
    if (city) params.append('city', city);
    params.append('limit', limit.toString());
    
    const response = await axios.get(
      `${API_BASE_URL}/api/rental-yield/top-localities?${params.toString()}`
    );
    return response.data;
  },

  getPropertyAnalysis: async (propertyId: number): Promise<PropertyRentalAnalysis> => {
    const response = await axios.get(
      `${API_BASE_URL}/api/rental-yield/property/${propertyId}`
    );
    return response.data;
  },

  triggerCalculation: async (): Promise<{ message: string }> => {
    const response = await axios.post(`${API_BASE_URL}/api/rental-yield/calculate`);
    return response.data;
  }
};
