const API_BASE_URL = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export interface RiskSubmission {
  property_address: string;
  builder_name?: string;
  project_name?: string;
  property_type: string;
  budget_range?: string;
}

export interface DocumentUpload {
  document_type: string;
  file: File;
}

export interface ReraCheck {
  rera_id?: string;
  project_name?: string;
  submission_id: string;
}

export interface OwnershipCheck {
  submission_id: string;
  owner_name: string;
  document_details?: Record<string, any>;
}

export interface EncumbranceCheck {
  submission_id: string;
  property_details?: Record<string, any>;
}

export interface RiskSubmissionResponse {
  id: string;
  property_address: string;
  builder_name?: string;
  project_name?: string;
  property_type: string;
  budget_range?: string;
  status: string;
  user_id: string;
  created_at: string;
  updated_at: string;
  documents: DocumentResponse[];
  rera_check?: ReraCheckResponse;
  ownership_check?: OwnershipCheckResponse;
  encumbrance_check?: EncumbranceCheckResponse;
  risk_score?: RiskScoreResponse;
  admin_review?: AdminReviewResponse;
}

export interface DocumentResponse {
  id: string;
  submission_id: string;
  document_type: string;
  file_url: string;
  file_name: string;
  file_size: number;
  status: string;
  uploaded_at: string;
  verified_at?: string;
}

export interface ReraCheckResponse {
  id: string;
  submission_id: string;
  rera_id?: string;
  project_name?: string;
  status: 'pass' | 'warning' | 'fail';
  details: Record<string, any>;
  checked_at: string;
}

export interface OwnershipCheckResponse {
  id: string;
  submission_id: string;
  owner_name: string;
  status: 'verified' | 'mismatch' | 'incomplete';
  details: Record<string, any>;
  checked_at: string;
}

export interface EncumbranceCheckResponse {
  id: string;
  submission_id: string;
  status: 'clear' | 'found' | 'pending';
  details: Record<string, any>;
  checked_at: string;
}

export interface RiskScoreResponse {
  id: string;
  submission_id: string;
  overall_score: number;
  risk_level: 'low' | 'medium' | 'high' | 'critical';
  document_score: number;
  rera_score: number;
  ownership_score: number;
  encumbrance_score: number;
  breakdown: Record<string, any>;
  recommendations: string[];
  calculated_at: string;
}

export interface AdminReviewResponse {
  id: string;
  submission_id: string;
  status: 'pending' | 'approved' | 'rejected' | 'resubmission_required';
  comments?: string;
  admin_id: string;
  reviewed_at: string;
}

export interface RiskReportResponse {
  submission: RiskSubmissionResponse;
  report_url?: string;
  generated_at?: string;
}

export interface OwnershipHistoryRecord {
  owner_name: string;
  transfer_date: string;
  transfer_type: string;
  previous_owner?: string;
  document_type: string;
  is_current_owner: boolean;
  verified: boolean;
}

export interface VerificationScore {
  property_id: string;
  overall_score: number;
  grade: string;
  risk_level: string;
  component_scores: {
    document_score: number;
    ownership_score: number;
    legal_score: number;
    rera_score: number;
    financial_score: number;
  };
  breakdown: Record<string, string>;
  risk_factors: string[];
  recommendations: string[];
  last_updated?: string;
}

class RiskComplianceService {
  private getAuthHeaders() {
    const token = localStorage.getItem('access_token');
    return {
      'Authorization': `Bearer ${token}`,
      'Content-Type': 'application/json'
    };
  }

  private getMultipartHeaders() {
    const token = localStorage.getItem('access_token');
    return {
      'Authorization': `Bearer ${token}`
    };
  }

  async createSubmission(data: RiskSubmission): Promise<{ success: boolean; submission_id: string }> {
    // Mock implementation for development
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          success: true,
          submission_id: `sub_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
        });
      }, 1000);
    });
  }

  async uploadDocument(submissionId: string, documentType: string, file: File): Promise<{ success: boolean; document_id: string; file_url: string }> {
    // Mock implementation
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          success: true,
          document_id: `doc_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
          file_url: `mock://documents/${file.name}`
        });
      }, 1500);
    });
  }

  async performReraCheck(data: ReraCheck): Promise<{ success: boolean; check_id: string }> {
    // Mock implementation for development
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          success: true,
          check_id: `rera_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
        });
      }, 1500);
    });
  }

  async performOwnershipCheck(data: OwnershipCheck): Promise<{ success: boolean; check_id: string }> {
    // Mock implementation for development
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          success: true,
          check_id: `owner_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
        });
      }, 1500);
    });
  }

  async performEncumbranceCheck(data: EncumbranceCheck): Promise<{ success: boolean; check_id: string }> {
    // Mock implementation for development
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          success: true,
          check_id: `encumb_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
        });
      }, 1500);
    });
  }

  async calculateRiskScore(submissionId: string): Promise<{ success: boolean; score_id: string }> {
    // Mock implementation for development
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          success: true,
          score_id: `score_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
        });
      }, 2000);
    });
  }

  async getSubmission(submissionId: string): Promise<RiskSubmissionResponse> {
    // Mock implementation for development
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          id: submissionId,
          property_address: "123 Sample Street, Bangalore",
          builder_name: "Sample Builders Ltd",
          project_name: "Sample Project",
          property_type: "apartment",
          budget_range: "50L-1Cr",
          status: "completed",
          user_id: "user123",
          created_at: new Date().toISOString(),
          updated_at: new Date().toISOString(),
          documents: [],
          rera_check: {
            id: "rera123",
            submission_id: submissionId,
            status: "pass",
            details: { validity: "Active" },
            checked_at: new Date().toISOString()
          },
          ownership_check: {
            id: "owner123",
            submission_id: submissionId,
            owner_name: "Sample Owner",
            status: "verified",
            details: {},
            checked_at: new Date().toISOString()
          },
          encumbrance_check: {
            id: "encumb123",
            submission_id: submissionId,
            status: "clear",
            details: {},
            checked_at: new Date().toISOString()
          }
        });
      }, 1000);
    });
  }

  async getUserSubmissions(): Promise<RiskSubmissionResponse[]> {
    const response = await fetch(`${API_BASE_URL}/risk/user-submissions`, {
      method: 'GET',
      headers: this.getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`Failed to get user submissions: ${response.statusText}`);
    }
    
    return response.json();
  }

  async getAdminSubmission(submissionId: string): Promise<RiskSubmissionResponse> {
    const response = await fetch(`${API_BASE_URL}/risk/admin/submissions/${submissionId}`, {
      method: 'GET',
      headers: this.getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`Failed to get admin submission: ${response.statusText}`);
    }
    
    return response.json();
  }

  async generateReport(submissionId: string): Promise<{ success: boolean; report_url: string; generated_at: string }> {
    // Mock implementation for development
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          success: true,
          report_url: `mock://reports/risk-compliance-${submissionId}.pdf`,
          generated_at: new Date().toISOString()
        });
      }, 2000);
    });
  }

  // Admin methods

  async adminReviewSubmission(submissionId: string, status: string, comments?: string): Promise<{ success: boolean; review_id: string }> {
    const response = await fetch(`${API_BASE_URL}/risk/review/${submissionId}`, {
      method: 'POST',
      headers: this.getAuthHeaders(),
      body: JSON.stringify({ 
        status: status === 'approved' ? 'approved' : 'rejected', 
        comments: comments || ''
      })
    });
    
    if (!response.ok) {
      const errorText = await response.text();
      console.error('Review submission failed:', errorText);
      throw new Error(`Failed to review submission: ${response.status} ${response.statusText}`);
    }
    
    return response.json();
  }

  async getAdminReviews(limit = 50, offset = 0): Promise<any[]> {
    const response = await fetch(`${API_BASE_URL}/risk/reviews?limit=${limit}&offset=${offset}`, {
      method: 'GET',
      headers: this.getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`Failed to get admin reviews: ${response.statusText}`);
    }
    
    return response.json();
  }

  async getOwnershipHistory(propertyId: string): Promise<OwnershipHistoryRecord[]> {
    const response = await fetch(`${API_BASE_URL}/risk/ownership-history/${propertyId}`, {
      method: 'GET',
      headers: this.getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`Failed to get ownership history: ${response.statusText}`);
    }
    
    const result = await response.json();
    return result.data || [];
  }

  async getOwnershipHistoryBySubmission(submissionId: string): Promise<OwnershipHistoryRecord[]> {
    const response = await fetch(`${API_BASE_URL}/risk/ownership-history/submission/${submissionId}`, {
      method: 'GET',
      headers: this.getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`Failed to get ownership history: ${response.statusText}`);
    }
    
    const result = await response.json();
    return result.data || [];
  }

  async getVerificationScore(submissionId: string): Promise<VerificationScore> {
    // Mock implementation for development
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          property_id: submissionId,
          overall_score: 85,
          grade: "A",
          risk_level: "low",
          component_scores: {
            document_score: 90,
            ownership_score: 85,
            legal_score: 80,
            rera_score: 95,
            financial_score: 75
          },
          breakdown: {
            "Document Verification": "Excellent",
            "Ownership Clarity": "Good",
            "Legal Compliance": "Good",
            "RERA Status": "Excellent",
            "Financial Health": "Fair"
          },
          risk_factors: ["Minor documentation gaps"],
          recommendations: [
            "Property shows excellent compliance",
            "Proceed with confidence",
            "Consider title insurance for additional protection"
          ],
          last_updated: new Date().toISOString()
        });
      }, 1500);
    });
  }

  async calculateVerificationScore(submissionId: string): Promise<VerificationScore> {
    const response = await fetch(`${API_BASE_URL}/risk/calculate-verification-score/${submissionId}`, {
      method: 'POST',
      headers: this.getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`Failed to calculate verification score: ${response.statusText}`);
    }
    
    const result = await response.json();
    return result.data;
  }

  async analyzeDocuments(submissionId: string): Promise<{ success: boolean; ownership_records: OwnershipHistoryRecord[] }> {
    const response = await fetch(`${API_BASE_URL}/risk/analyze-documents/${submissionId}`, {
      method: 'POST',
      headers: this.getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`Failed to analyze documents: ${response.statusText}`);
    }
    
    return response.json();
  }

  async analyzeTitleChain(submissionId: string): Promise<{ success: boolean; data: any }> {
    // Mock implementation
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          success: true,
          data: {
            integrity_score: 85,
            total_transfers: 4,
            verified_transfers: 3,
            gaps_found: 1,
            risk_level: 'low',
            gaps: [{
              type: 'documentation_gap',
              description: 'Minor documentation discrepancy in transfer from 2019',
              severity: 'low'
            }],
            recommendations: [
              'Verify documentation for 2019 transfer',
              'Obtain certified copies of missing documents',
              'Consider title insurance for additional protection'
            ]
          }
        });
      }, 2000);
    });
  }

  async getTitleChainAnalysis(submissionId: string): Promise<{ success: boolean; data: any }> {
    const response = await fetch(`${API_BASE_URL}/risk/title-chain-analysis/${submissionId}`, {
      method: 'GET',
      headers: this.getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`Failed to get title chain analysis: ${response.statusText}`);
    }
    
    return response.json();
  }

  async performLitigationCheck(submissionId: string): Promise<{ success: boolean; litigation_id: string }> {
    const response = await fetch(`${API_BASE_URL}/risk/litigation-check/${submissionId}`, {
      method: 'POST',
      headers: this.getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`Failed to perform litigation check: ${response.statusText}`);
    }
    
    return response.json();
  }

  async getLitigationReport(submissionId: string): Promise<{ success: boolean; data: any }> {
    const response = await fetch(`${API_BASE_URL}/risk/litigation-report/${submissionId}`, {
      method: 'GET',
      headers: this.getAuthHeaders()
    });
    
    if (!response.ok) {
      throw new Error(`Failed to get litigation report: ${response.statusText}`);
    }
    
    return response.json();
  }

  async analyzeLitigation(submissionId: string): Promise<{ success: boolean; data: any }> {
    // Mock implementation
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          success: true,
          data: {
            litigation_score: 92,
            total_cases: 0,
            active_cases: 0,
            resolved_cases: 0,
            risk_level: 'low',
            cases: [],
            recommendations: [
              'No litigation history found - excellent legal standing',
              'Continue monitoring for any future legal disputes',
              'Maintain proper documentation for future reference'
            ]
          }
        });
      }, 1500);
    });
  }
}

export const riskComplianceService = new RiskComplianceService();