const API_BASE = import.meta.env.VITE_API_BASE || '/api';

export interface SearchHistoryEntry {
  id?: number;
  user_id: string;
  query: string;
  filters?: {
    property_type?: string[];
    bedrooms?: number;
    city?: string;
    listing_type?: string[];
    min_price?: number;
    max_price?: number;
    furnishing?: string[];
    amenities?: string[];
  };
  location?: string;
  timestamp?: string;
}

export interface SearchRecommendation {
  id: number;
  title: string;
  location: string;
  beds: string;
  baths: string;
  area: string;
  price: string;
  imageUrl?: string;
  urgent?: boolean;
  property_type: string;
  city: string;
}

export const searchHistoryAPI = {
  async storeSearch(data: SearchHistoryEntry): Promise<SearchHistoryEntry> {
    const response = await fetch(`${API_BASE}/search-history/`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(data)
    });
    if (!response.ok) throw new Error('Failed to store search');
    return response.json();
  },

  async getUserHistory(userId: string, limit = 10): Promise<SearchHistoryEntry[]> {
    const response = await fetch(`${API_BASE}/search-history/${userId}?limit=${limit}`);
    if (!response.ok) throw new Error('Failed to fetch search history');
    return response.json();
  },

  async getUserSearchCount(userId: string): Promise<number> {
    try {
      const response = await fetch(`${API_BASE}/search-history/${userId}/count`);
      if (!response.ok) return 0;
      const data = await response.json();
      return data.total_count || 0;
    } catch (error) {
      console.error('Error fetching search count:', error);
      return 0;
    }
  },

  async getRecommendations(userId: string): Promise<SearchRecommendation[]> {
    const response = await fetch(`${API_BASE}/search-history/recommendations/${userId}`);
    if (!response.ok) throw new Error('Failed to fetch recommendations');
    return response.json();
  },

  async clearAllHistory(userId: string): Promise<void> {
    const response = await fetch(`${API_BASE}/search-history/${userId}`, {
      method: 'DELETE'
    });
    if (!response.ok) throw new Error('Failed to clear search history');
  }
};