const API_BASE_URL = 'http://localhost:8090';

export interface SellerPropertyStats {
  total_properties: number;
  active_properties: number;
  sold_properties: number;
  rented_properties: number;
  pending_properties: number;
  rejected_properties: number;
  properties_by_type: Record<string, number>;
  properties_by_listing_type: Record<string, number>;
  properties_by_city: Record<string, number>;
  average_price: number;
  total_views: number;
  total_inquiries: number;
  new_properties_today: number;
  new_properties_this_week: number;
  new_properties_this_month: number;
  absorption_rate: number;
}

export const sellerDashboardAPI = {
  /**
   * Get seller-specific property statistics with strict data isolation.
   * Only returns data for properties owned by the authenticated seller.
   */
  getSellerPropertyStats: async (userId: string): Promise<SellerPropertyStats> => {
    const timestamp = new Date().getTime();
    const token = localStorage.getItem('token') || sessionStorage.getItem('token');
    
    const headers: Record<string, string> = {
      'Cache-Control': 'no-cache',
      'Pragma': 'no-cache',
      'Content-Type': 'application/json'
    };
    
    if (token) {
      headers['Authorization'] = `Bearer ${token}`;
    }
    
    const response = await fetch(
      `${API_BASE_URL}/api/seller-dashboard/properties?user_id=${userId}&_t=${timestamp}`,
      {
        cache: 'no-store',
        headers
      }
    );
    
    if (!response.ok) {
      throw new Error(`Failed to fetch seller property stats: ${response.status}`);
    }
    
    return await response.json();
  }
};