const API_BASE = import.meta.env.VITE_API_BASE || 'http://localhost:8090/api';

export interface PropertyMetrics {
  total_views: number;
  total_saves: number;
  total_visits: number;
  total_wishlist: number;
  unique_users: number;
  unique_sessions: number;
  engagement_score: number;
}

export interface PropertyAnalytics {
  property_id: number;
  metrics: PropertyMetrics;
  conversion_rates: {
    view_to_save: number;
    save_to_visit: number;
  };
  time_series: Array<{
    date: string;
    views: number;
    saves: number;
  }>;
  days_range: number;
}

export interface PropertySummary {
  property_id: number;
  title: string;
  location: string;
  property_type: string;
  price: number;
  status: string;
  metrics: {
    views: number;
    saves: number;
    visits: number;
    wishlist: number;
    unique_users: number;
    engagement_score: number;
  };
}

export interface PortfolioAnalytics {
  total_properties: number;
  properties: PropertySummary[];
  summary: {
    total_views: number;
    total_saves: number;
    total_visits: number;
    total_wishlist: number;
    unique_users: number;
  };
  days_range: number;
}

export interface EngagedUser {
  user_id: string;
  email?: string;
  phone?: string;
  name: string;
  activity: {
    views: number;
    saves: number;
    visits: number;
    wishlist: number;
  };
  engagement_level: 'high' | 'medium' | 'low';
  engagement_score: number;
  segment: string;
  last_interaction: string;
  first_interaction: string;
}

export interface PropertyEngagement {
  property_id: number;
  total_engaged_users: number;
  users: EngagedUser[];
}

export interface PropertyInsights {
  property_id: number;
  peak_hours: Array<{
    hour: number;
    activity: number;
  }>;
  geographic_distribution: Array<{
    location: string;
    users: number;
  }>;
  market_comparison: {
    current_activity: number;
    average_activity: number;
    market_position: string;
    similar_properties_count: number;
  };
}

const getAuthHeaders = () => {
  const token = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
  return {
    'Authorization': `Bearer ${token}`,
    'Content-Type': 'application/json'
  };
};

export const sellerBehaviorApi = {
  async getPropertyAnalytics(propertyId: number, days: number = 30): Promise<PropertyAnalytics> {
    const response = await fetch(
      `${API_BASE}/behavior/seller/property-analytics/${propertyId}?days=${days}`,
      { headers: getAuthHeaders() }
    );
    if (!response.ok) throw new Error('Failed to fetch property analytics');
    return response.json();
  },

  async getMyPropertiesAnalytics(days: number = 30): Promise<PortfolioAnalytics> {
    const response = await fetch(
      `${API_BASE}/behavior/seller/my-properties-analytics?days=${days}`,
      { headers: getAuthHeaders() }
    );
    if (!response.ok) throw new Error('Failed to fetch portfolio analytics');
    return response.json();
  },

  async getPropertyEngagement(propertyId: number, limit: number = 50): Promise<PropertyEngagement> {
    const response = await fetch(
      `${API_BASE}/behavior/seller/property-engagement/${propertyId}?limit=${limit}`,
      { headers: getAuthHeaders() }
    );
    if (!response.ok) throw new Error('Failed to fetch property engagement');
    return response.json();
  },

  async getPropertyInsights(propertyId: number): Promise<PropertyInsights> {
    const response = await fetch(
      `${API_BASE}/behavior/seller/property-insights/${propertyId}`,
      { headers: getAuthHeaders() }
    );
    if (!response.ok) throw new Error('Failed to fetch property insights');
    return response.json();
  }
};
